<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // Ensure both tables exist before adding FKs
        if (!Schema::hasTable('items') || !Schema::hasTable('users') || !Schema::hasTable('gs_adjustments')) {
            // If any table missing, skip safely
            return;
        }

        // Read column types from information_schema
        $dbName = DB::getDatabaseName();

        $itemsId = DB::table('information_schema.COLUMNS')
            ->select('COLUMN_TYPE', 'DATA_TYPE')
            ->where('TABLE_SCHEMA', $dbName)
            ->where('TABLE_NAME', 'items')
            ->where('COLUMN_NAME', 'id')
            ->first();

        $usersId = DB::table('information_schema.COLUMNS')
            ->select('COLUMN_TYPE', 'DATA_TYPE')
            ->where('TABLE_SCHEMA', $dbName)
            ->where('TABLE_NAME', 'users')
            ->where('COLUMN_NAME', 'id')
            ->first();

        // Defensive: also ensure engines are InnoDB
        $engines = DB::select("
            SELECT TABLE_NAME, ENGINE
            FROM information_schema.TABLES
            WHERE TABLE_SCHEMA = ? AND TABLE_NAME IN ('items','users','gs_adjustments')
        ", [$dbName]);

        foreach ($engines as $row) {
            if (strtolower($row->ENGINE ?? '') !== 'innodb') {
                // Convert engine if needed (safe for empty/brand new db)
                DB::statement("ALTER TABLE `{$row->TABLE_NAME}` ENGINE=InnoDB");
            }
        }

        // Align gs_adjustments.item_id type with items.id
        if ($itemsId) {
            $itemsIsBigint = stripos($itemsId->COLUMN_TYPE, 'bigint') === 0;
            $itemsUnsigned = stripos($itemsId->COLUMN_TYPE, 'unsigned') !== false;

            if ($itemsIsBigint) {
                // Make sure item_id is BIGINT [UNSIGNED] to match items.id
                DB::statement("ALTER TABLE `gs_adjustments` MODIFY `item_id` BIGINT" . ($itemsUnsigned ? " UNSIGNED" : "") . " NOT NULL");
            } else {
                // Assume INT family (int/mediumint/smallint rarely used for PK; int is common)
                // Match unsigned-ness
                $unsigned = $itemsUnsigned ? " UNSIGNED" : "";
                DB::statement("ALTER TABLE `gs_adjustments` MODIFY `item_id` INT{$unsigned} NOT NULL");
            }
        }

        // Align gs_adjustments.adjusted_by with users.id type
        if ($usersId) {
            $usersIsBigint = stripos($usersId->COLUMN_TYPE, 'bigint') === 0;
            $usersUnsigned = stripos($usersId->COLUMN_TYPE, 'unsigned') !== false;

            if ($usersIsBigint) {
                DB::statement("ALTER TABLE `gs_adjustments` MODIFY `adjusted_by` BIGINT" . ($usersUnsigned ? " UNSIGNED" : "") . " NULL");
            } else {
                $unsigned = $usersUnsigned ? " UNSIGNED" : "";
                DB::statement("ALTER TABLE `gs_adjustments` MODIFY `adjusted_by` INT{$unsigned} NULL");
            }
        }

        // Now add FKs (drop if already present)
        $this->dropFkIfExists('gs_adjustments', 'gs_adj_item_fk');
        $this->dropFkIfExists('gs_adjustments', 'gs_adj_by_fk');

        // Add FK for item_id -> items(id)
        DB::statement("
            ALTER TABLE `gs_adjustments`
            ADD CONSTRAINT `gs_adj_item_fk`
            FOREIGN KEY (`item_id`) REFERENCES `items`(`id`)
            ON UPDATE CASCADE
            ON DELETE RESTRICT
        ");

        // Add FK for adjusted_by -> users(id), set null on delete
        DB::statement("
            ALTER TABLE `gs_adjustments`
            ADD CONSTRAINT `gs_adj_by_fk`
            FOREIGN KEY (`adjusted_by`) REFERENCES `users`(`id`)
            ON UPDATE CASCADE
            ON DELETE SET NULL
        ");
    }

    public function down(): void
    {
        // Drop FKs if exist
        $this->dropFkIfExists('gs_adjustments', 'gs_adj_item_fk');
        $this->dropFkIfExists('gs_adjustments', 'gs_adj_by_fk');
    }

    /**
     * Safely drop a foreign key if it exists.
     */
    private function dropFkIfExists(string $table, string $fkName): void
    {
        $dbName = DB::getDatabaseName();

        $exists = DB::table('information_schema.TABLE_CONSTRAINTS')
            ->where('TABLE_SCHEMA', $dbName)
            ->where('TABLE_NAME', $table)
            ->where('CONSTRAINT_NAME', $fkName)
            ->where('CONSTRAINT_TYPE', 'FOREIGN KEY')
            ->exists();

        if ($exists) {
            DB::statement("ALTER TABLE `{$table}` DROP FOREIGN KEY `{$fkName}`");
        }
    }
};
