<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // 1) Create table and columns first (no FKs yet)
        Schema::create('stocks', function (Blueprint $table) {
            $table->engine = 'InnoDB';

            $table->id();

            // FK columns (same type as parents: BIGINT UNSIGNED)
            $table->unsignedBigInteger('item_id');
            $table->unsignedBigInteger('created_by');
            $table->unsignedBigInteger('updated_by')->nullable();

            // Other columns (match your dump)
            $table->timestamp('date')->useCurrent()->useCurrentOnUpdate();
            $table->string('type', 191);
            $table->integer('source_id');              // INT(11)
            $table->string('source_number', 191);

            $table->decimal('debit_qty',  15, 3)->default(0);
            $table->decimal('credit_qty', 15, 3)->default(0);
            $table->decimal('debit_rate', 15, 3)->default(0);
            $table->decimal('credit_rate',15, 3)->default(0);

            $table->timestamps();

            // Helpful indexes
            $table->index('item_id');
            $table->index('created_by');
            $table->index('updated_by');
            $table->index(['type', 'source_id', 'source_number']);
        });

        // 2) Add FKs only if parent tables exist (prevents errno 150)
        Schema::table('stocks', function (Blueprint $table) {
            if (Schema::hasTable('items')) {
                $table->foreign('item_id')
                      ->references('id')->on('items')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();
            }

            if (Schema::hasTable('users')) {
                $table->foreign('created_by')
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();

                $table->foreign('updated_by')
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }
        });
    }

    public function down(): void
    {
        // Drop FKs if they exist, then drop table
        Schema::table('stocks', function (Blueprint $table) {
            // Guarded drops so it doesn’t error if FKs were never added
            foreach (['stocks_item_id_foreign', 'stocks_created_by_foreign', 'stocks_updated_by_foreign'] as $fk) {
                try { $table->dropForeign($fk); } catch (\Throwable $e) { /* ignore */ }
            }
        });

        Schema::dropIfExists('stocks');
    }
};


