<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration
{
    // Possible parent tables for group_id (first existing wins)
    private array $groupParents = ['groups', 'season_groups', 'item_groups'];

    private function pickExisting(array $candidates): ?string
    {
        foreach ($candidates as $t) {
            if (Schema::hasTable($t)) return $t;
        }
        return null;
    }

    private function fkExists(string $table, string $name): bool
    {
        $sql = <<<SQL
SELECT 1
FROM information_schema.TABLE_CONSTRAINTS
WHERE TABLE_SCHEMA = DATABASE()
  AND TABLE_NAME = ?
  AND CONSTRAINT_TYPE = 'FOREIGN KEY'
  AND CONSTRAINT_NAME = ?
LIMIT 1
SQL;
        return !empty(DB::select($sql, [$table, $name]));
    }

    public function up(): void
    {
        $groupParent = $this->pickExisting($this->groupParents);
        if (!$groupParent) {
            // Adjust this to your real parent table if different
            throw new RuntimeException(
                "No parent table found for group_id. Tried: ".implode(', ', $this->groupParents).
                ". Create one (with an 'id' PK) or edit the migration to the correct table."
            );
        }
        if (!Schema::hasTable('users')) {
            throw new RuntimeException("Required parent table 'users' not found for created_by/updated_by.");
        }

        if (!Schema::hasTable('seasons')) {
            // Create fresh table
            Schema::create('seasons', function (Blueprint $table) use ($groupParent) {
                $table->engine = 'InnoDB';

                $table->id();                                // bigint unsigned AI
                $table->string('name', 191);                 // NOT NULL

                // Required FKs (NOT NULL as per your dump)
                $table->foreignId('group_id')
                      ->constrained($groupParent)
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();

                $table->foreignId('created_by')
                      ->constrained('users')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();

                $table->foreignId('updated_by')
                      ->constrained('users')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();

                $table->timestamps();                        // nullable created_at / updated_at
            });
        } else {
            // Table exists (from dump). Ensure FKs are present.
            Schema::table('seasons', function (Blueprint $table) use ($groupParent) {
                if (Schema::hasColumn('seasons', 'group_id') &&
                    !$this->fkExists('seasons', 'seasons_group_id_foreign')) {
                    $table->foreign('group_id', 'seasons_group_id_foreign')
                          ->references('id')->on($groupParent)
                          ->cascadeOnUpdate()
                          ->restrictOnDelete();
                }

                if (Schema::hasColumn('seasons', 'created_by') &&
                    !$this->fkExists('seasons', 'seasons_created_by_foreign')) {
                    $table->foreign('created_by', 'seasons_created_by_foreign')
                          ->references('id')->on('users')
                          ->cascadeOnUpdate()
                          ->restrictOnDelete();
                }

                if (Schema::hasColumn('seasons', 'updated_by') &&
                    !$this->fkExists('seasons', 'seasons_updated_by_foreign')) {
                    $table->foreign('updated_by', 'seasons_updated_by_foreign')
                          ->references('id')->on('users')
                          ->cascadeOnUpdate()
                          ->restrictOnDelete();
                }
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('seasons');
    }
};

