<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {

    /** Check if a FK with this name exists on a table */
    private function fkExists(string $table, string $name): bool
    {
        $sql = <<<SQL
SELECT 1
FROM information_schema.TABLE_CONSTRAINTS
WHERE TABLE_SCHEMA = DATABASE()
  AND TABLE_NAME = ?
  AND CONSTRAINT_TYPE = 'FOREIGN KEY'
  AND CONSTRAINT_NAME = ?
LIMIT 1
SQL;
        return !empty(DB::select($sql, [$table, $name]));
    }

    public function up(): void
    {
        // 1) Create the table only if it doesn't exist
        if (!Schema::hasTable('rm_warehouse_types')) {
            Schema::create('rm_warehouse_types', function (Blueprint $t) {
                $t->engine = 'InnoDB';

                $t->id();
                $t->string('name', 191);
                $t->string('status', 191)->default('1');

                $t->unsignedBigInteger('created_by')->nullable();
                $t->unsignedBigInteger('updated_by')->nullable();

                $t->timestamp('created_at')->nullable();
                $t->timestamp('updated_at')->nullable();
            });
        }

        // 2) Add FKs only if `users` exists and the constraints are not already present
        if (Schema::hasTable('users')) {
            Schema::table('rm_warehouse_types', function (Blueprint $t) {
                $createdFk = 'rm_warehouse_types_created_by_foreign';
                $updatedFk = 'rm_warehouse_types_updated_by_foreign';

                if (!$this->fkExists($t->getTable(), $createdFk) && Schema::hasColumn($t->getTable(), 'created_by')) {
                    $t->foreign('created_by', $createdFk)
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
                }

                if (!$this->fkExists($t->getTable(), $updatedFk) && Schema::hasColumn($t->getTable(), 'updated_by')) {
                    $t->foreign('updated_by', $updatedFk)
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
                }
            });
        }
    }

    public function down(): void
    {
        Schema::dropIfExists('rm_warehouse_types');
    }
};


