<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        Schema::create('rm_transactions', function (Blueprint $t) {
            $t->engine = 'InnoDB';

            $t->id(); // BIGINT UNSIGNED AI

            // Columns from your dump
            $t->unsignedBigInteger('company_id')->nullable();      // FK -> companies.id (nullable)
            $t->string('account_type', 191)->nullable();
            $t->unsignedBigInteger('account_id')->nullable();      // kept without FK (none in dump)
            $t->string('invoice_no', 191)->nullable();
            $t->string('reference', 191)->nullable();

            // Polymorphic target (kept as plain columns to match dump)
            $t->string('transactionable_type', 191)->nullable();
            $t->unsignedBigInteger('transactionable_id')->nullable();

            $t->string('date', 191)->nullable();                   // string in dump (not DATE)
            $t->string('description', 191)->nullable();
            $t->string('balance_type', 191)->nullable();

            $t->decimal('debit_amount', 10, 2)->nullable();
            $t->decimal('credit_amount', 10, 2)->nullable();
            $t->decimal('balance_amount', 10, 2)->nullable();

            $t->unsignedBigInteger('created_by');                  // NOT NULL in dump
            $t->unsignedBigInteger('updated_by')->nullable();

            // Timestamps are nullable in dump
            $t->timestamp('created_at')->nullable();
            $t->timestamp('updated_at')->nullable();

            // Optional: index to speed up polymorphic lookups
            $t->index(['transactionable_type', 'transactionable_id'], 'rm_transactions_transactionable_idx');
        });

        // Add FKs only if parent tables exist (prevents errno:150)
        Schema::table('rm_transactions', function (Blueprint $t) {
            if (Schema::hasTable('companies')) {
                $t->foreign('company_id', 'rm_transactions_company_id_foreign')
                  ->references('id')->on('companies')
                  ->cascadeOnUpdate()
                  ->nullOnDelete();
            }

            if (Schema::hasTable('users')) {
                $t->foreign('created_by', 'rm_transactions_created_by_foreign')
                  ->references('id')->on('users')
                  ->cascadeOnUpdate()
                  ->restrictOnDelete();

                $t->foreign('updated_by', 'rm_transactions_updated_by_foreign')
                  ->references('id')->on('users')
                  ->cascadeOnUpdate()
                  ->nullOnDelete();
            }
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('rm_transactions');
    }
};
