<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        Schema::create('rm_sales', function (Blueprint $table) {
            $table->engine = 'InnoDB';

            $table->id();

            // Columns (exactly as your dump)
            $table->unsignedBigInteger('company_id')->nullable()->index('rm_sales_company_id_index');
            $table->string('invoice_no', 191)->nullable();

            $table->unsignedBigInteger('customer_id')->nullable()->index('rm_sales_customer_id_index');
            $table->unsignedBigInteger('employee_id')->nullable()->index('rm_sales_employee_id_index');

            $table->date('date')->nullable();

            $table->decimal('subtotal',       16, 2)->nullable();
            $table->decimal('vat',            16, 2)->nullable();
            $table->decimal('discount',       16, 2)->nullable();
            $table->decimal('shipping_cost',  16,  2)->nullable();
            $table->decimal('payable_amount', 16,  2)->nullable();
            $table->decimal('paid_amount',    16,  2)->nullable();
            $table->decimal('due_amount',     16,  2)->nullable();

            $table->string('reference', 191)->nullable();

            $table->unsignedBigInteger('currency')->nullable();
            $table->unsignedBigInteger('payment_type')->nullable();

            $table->tinyInteger('delivery_status')->default(0);

            $table->unsignedBigInteger('created_by')->index('rm_sales_created_by_index');
            $table->unsignedBigInteger('updated_by')->nullable()->index('rm_sales_updated_by_index');

            $table->timestamps();

            $table->unsignedBigInteger('warehouse_id')->nullable();
            $table->decimal('previous_due', 16, 2)->nullable();
            $table->decimal('sale_cost',    16, 2)->nullable();
        });

        // ---- Add FKs only if parent tables exist (avoids errno:150) ----
        Schema::table('rm_sales', function (Blueprint $table) {
            // company_id -> companies.id
            if (Schema::hasTable('companies')) {
                $table->foreign('company_id', 'rm_sales_company_id_foreign')
                      ->references('id')->on('companies')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // customer_id -> rm_customers.id (fallback: customers)
            if (Schema::hasTable('rm_customers')) {
                $table->foreign('customer_id', 'rm_sales_customer_id_foreign')
                      ->references('id')->on('rm_customers')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            } elseif (Schema::hasTable('customers')) {
                $table->foreign('customer_id', 'rm_sales_customer_id_foreign')
                      ->references('id')->on('customers')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // employee_id -> rm_employees.id (fallback: employees)
            if (Schema::hasTable('rm_employees')) {
                $table->foreign('employee_id', 'rm_sales_employee_id_foreign')
                      ->references('id')->on('rm_employees')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            } elseif (Schema::hasTable('employees')) {
                $table->foreign('employee_id', 'rm_sales_employee_id_foreign')
                      ->references('id')->on('employees')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // created_by / updated_by -> users.id
            if (Schema::hasTable('users')) {
                $table->foreign('created_by', 'rm_sales_created_by_foreign')
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();

                $table->foreign('updated_by', 'rm_sales_updated_by_foreign')
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // warehouse_id -> warehouses.id
            if (Schema::hasTable('warehouses')) {
                $table->foreign('warehouse_id', 'rm_sales_warehouse_id_foreign')
                      ->references('id')->on('warehouses')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // currency -> currencies.id  (optional table)
            if (Schema::hasTable('currencies')) {
                $table->foreign('currency', 'rm_sales_currency_foreign')
                      ->references('id')->on('currencies')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // payment_type -> payment_types.id  (optional table)
            if (Schema::hasTable('payment_types')) {
                $table->foreign('payment_type', 'rm_sales_payment_type_foreign')
                      ->references('id')->on('payment_types')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('rm_sales');
    }
};
