<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // 1) Create table (no FKs yet)
        Schema::create('rm_product_ledgers', function (Blueprint $table) {
            $table->engine = 'InnoDB';

            $table->id(); // BIGINT UNSIGNED AI

            // polymorphic source (as in dump)
            $table->string('sourceable_type', 191);
            $table->unsignedBigInteger('sourceable_id');

            // columns from your dump
            $table->unsignedBigInteger('product_id');         // NOT NULL
            $table->unsignedBigInteger('unit_id')->nullable();
            $table->unsignedBigInteger('company_id')->nullable();
            $table->date('date')->nullable();

            // Note: 'in' and 'out' are reserved words; Laravel will quote them.
            $table->decimal('quantity', 10, 2)->nullable();
            $table->decimal('in',       10, 2)->nullable();
            $table->decimal('out',      10, 2)->nullable();
            $table->decimal('current_quantity', 10, 2)->nullable();

            $table->string('remarks', 191)->nullable();

            $table->unsignedBigInteger('created_by');         // NOT NULL
            $table->unsignedBigInteger('updated_by')->nullable();

            $table->timestamp('created_at')->nullable();
            $table->timestamp('updated_at')->nullable();

            $table->unsignedBigInteger('warehouse_id')->nullable();

            // helpful indexes like in dump
            $table->index('product_id',  'rm_product_ledgers_product_id_index');
            $table->index('company_id',  'rm_product_ledgers_company_id_index');
            $table->index('created_by',  'rm_product_ledgers_created_by_index');
            $table->index('updated_by',  'rm_product_ledgers_updated_by_index');
            $table->index(['sourceable_type', 'sourceable_id'], 'rm_product_ledgers_sourceable_index');
        });

        // 2) Add FKs only if parents exist (prevents errno:150)
        Schema::table('rm_product_ledgers', function (Blueprint $table) {
            // product_id -> prefer rm_products, fall back to products
            if (Schema::hasTable('rm_products')) {
                $table->foreign('product_id', 'rm_product_ledgers_product_id_foreign')
                      ->references('id')->on('rm_products')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete(); // product_id is NOT NULL in dump
            } elseif (Schema::hasTable('products')) {
                $table->foreign('product_id', 'rm_product_ledgers_product_id_foreign')
                      ->references('id')->on('products')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();
            }

            // unit_id -> item_units (nullable)
            if (Schema::hasTable('item_units')) {
                $table->foreign('unit_id', 'rm_product_ledgers_unit_id_foreign')
                      ->references('id')->on('item_units')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // company_id -> companies (nullable)
            if (Schema::hasTable('companies')) {
                $table->foreign('company_id', 'rm_product_ledgers_company_id_foreign')
                      ->references('id')->on('companies')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // warehouse_id -> warehouses (nullable)
            if (Schema::hasTable('warehouses')) {
                $table->foreign('warehouse_id', 'rm_product_ledgers_warehouse_id_foreign')
                      ->references('id')->on('warehouses')
                      ->cascadeOnUpdate()
                      ->nullOnDelete();
            }

            // created_by / updated_by -> users
            if (Schema::hasTable('users')) {
                $table->foreign('created_by', 'rm_product_ledgers_created_by_foreign')
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->restrictOnDelete(); // NOT NULL

                $table->foreign('updated_by', 'rm_product_ledgers_updated_by_foreign')
                      ->references('id')->on('users')
                      ->cascadeOnUpdate()
                      ->nullOnDelete(); // nullable
            }
        });
    }

    public function down(): void
    {
        Schema::dropIfExists('rm_product_ledgers');
    }
};
