<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\DB;

return new class extends Migration {

    /** pick first existing table name from a list */
    private function pick(array $candidates): ?string {
        foreach ($candidates as $t) {
            if (Schema::hasTable($t)) return $t;
        }
        return null;
    }

    /** is table InnoDB? */
    private function isInnoDB(string $table): bool {
        $row = DB::selectOne("
            SELECT ENGINE FROM information_schema.TABLES
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?
            LIMIT 1
        ", [$table]);
        return $row && strtoupper($row->ENGINE ?? '') === 'INNODB';
    }

    /** does referenced column look like BIGINT UNSIGNED? */
    private function isBigintUnsigned(string $table, string $column = 'id'): bool {
        $row = DB::selectOne("
            SELECT DATA_TYPE, COLUMN_TYPE
            FROM information_schema.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
            LIMIT 1
        ", [$table, $column]);
        if (!$row) return false;
        $data = strtolower($row->DATA_TYPE ?? '');
        $ctype = strtolower($row->COLUMN_TYPE ?? '');
        return $data === 'bigint' && str_contains($ctype, 'unsigned');
        // (Laravel default: bigint(20) unsigned)
    }

    /** add FK iff parent exists, is InnoDB, and PK is BIGINT UNSIGNED */
    private function addForeignIfPossible(
        string $childTable,
        string $childCol,
        string $parentTable,
        string $parentCol,
        string $fkName,
        bool $nullable = true,
        string $onDelete = 'set null',
        string $onUpdate = 'cascade'
    ): void {
        if (!Schema::hasTable($parentTable)) return;
        if (!$this->isInnoDB($parentTable)) return;
        if (!$this->isBigintUnsigned($parentTable, $parentCol)) return;
        if (!Schema::hasColumn($childTable, $childCol)) return;

        Schema::table($childTable, function (Blueprint $t) use ($childCol, $parentTable, $parentCol, $fkName, $nullable, $onDelete, $onUpdate) {
            // name it explicitly to be predictable
            $foreign = $t->foreign($childCol, $fkName)->references($parentCol)->on($parentTable);
            // on update
            if ($onUpdate === 'cascade')   $foreign->cascadeOnUpdate();
            // on delete
            if ($nullable && $onDelete === 'set null') $foreign->nullOnDelete();
            elseif ($onDelete === 'restrict')          $foreign->restrictOnDelete();
            elseif ($onDelete === 'cascade')           $foreign->cascadeOnDelete();
        });
    }

    public function up(): void
    {
        // ---------- 1) Create table WITHOUT FKs (always succeeds) ----------
        Schema::create('rm_products', function (Blueprint $table) {
            $table->engine = 'InnoDB'; // ensure InnoDB
            $table->id();

            $table->string('name', 191);

            // FK-like columns
            $table->unsignedBigInteger('category_id');          // NOT NULL in dump
            $table->unsignedBigInteger('unit_id')->nullable();
            $table->unsignedBigInteger('supplier_id')->nullable();

            // other bigints (no FK in your dump)
            $table->unsignedBigInteger('small_unit_id')->nullable();
            $table->unsignedBigInteger('big_unit_id')->nullable();
            $table->unsignedBigInteger('warehouse_id')->nullable();
            $table->unsignedBigInteger('bag_warehouse_id')->nullable();

            $table->tinyInteger('type')->comment('1=>raw,2=finish');

            $table->decimal('purchase_price',   10, 2)->nullable();
            $table->decimal('sale_price',       10, 2)->nullable();
            $table->decimal('opening_quantity', 10, 2)->nullable();
            $table->decimal('alert_quantity',   10, 2)->nullable();
            $table->decimal('vat',              10, 2)->nullable();
            $table->decimal('conversion_to_kg', 10, 2)->nullable();

            $table->text('description')->nullable();
            $table->text('image')->nullable();

            $table->tinyInteger('status')->default(0);
            $table->decimal('bag_opening_quantity', 8, 2)->default(0.00);
            $table->tinyInteger('stock_type')->default(1);
            $table->decimal('product_price', 8, 2)->default(0.00);
            $table->decimal('bag_price',     8, 2)->default(0.00);

            // creator/updater (dump shows FK indexes to users)
            $table->unsignedBigInteger('created_by');
            $table->unsignedBigInteger('updated_by')->nullable();

            $table->timestamp('created_at')->nullable();
            $table->timestamp('updated_at')->nullable();
            $table->softDeletes(); // <-- added deleted_at

            // Helpful indexes
            $table->index('category_id', 'rm_products_category_id_index');
            $table->index('unit_id',     'rm_products_unit_id_index');
            $table->index('supplier_id', 'rm_products_supplier_id_index');
            $table->index('created_by',  'rm_products_created_by_index');
            $table->index('updated_by',  'rm_products_updated_by_index');
        });

        // ---------- 2) Add FKs ONLY if parent tables are OK ----------
        // category_id -> rm_categories(id) (restrict on delete, cascade on update)
        if (Schema::hasTable('rm_products')) {
            if ($this->isInnoDB('rm_products')) {
                $this->addForeignIfPossible(
                    'rm_products', 'category_id',
                    'rm_categories', 'id',
                    'rm_products_category_id_foreign',
                    false, // NOT NULL
                    'restrict', 'cascade'
                );

                // unit_id -> item_units(id) (nullable, set null on delete)
                $this->addForeignIfPossible(
                    'rm_products', 'unit_id',
                    'item_units', 'id',
                    'rm_products_unit_id_foreign',
                    true, 'set null', 'cascade'
                );

                // supplier_id -> suppliers|rm_suppliers(id) (nullable, set null on delete)
                $supplierParent = $this->pick(['suppliers', 'rm_suppliers']);
                if ($supplierParent) {
                    $this->addForeignIfPossible(
                        'rm_products', 'supplier_id',
                        $supplierParent, 'id',
                        'rm_products_supplier_id_foreign',
                        true, 'set null', 'cascade'
                    );
                }

                // created_by / updated_by -> users(id)
                $this->addForeignIfPossible(
                    'rm_products', 'created_by',
                    'users', 'id',
                    'rm_products_created_by_foreign',
                    false, 'restrict', 'cascade'
                );
                $this->addForeignIfPossible(
                    'rm_products', 'updated_by',
                    'users', 'id',
                    'rm_products_updated_by_foreign',
                    true, 'set null', 'cascade'
                );
            }
        }
    }

    public function down(): void
    {
        // Drop table (FKs will drop with it, but safe either way)
        Schema::dropIfExists('rm_products');
    }
};

