<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    private string $table = 'rm_bag_ledgers';

    private function hasForeignOn(string $col): bool
    {
        $sql = "SELECT 1 FROM information_schema.KEY_COLUMN_USAGE
                WHERE TABLE_SCHEMA = DATABASE()
                  AND TABLE_NAME   = ?
                  AND COLUMN_NAME  = ?
                  AND REFERENCED_TABLE_NAME IS NOT NULL
                LIMIT 1";
        return !empty(DB::select($sql, [$this->table, $col]));
    }

    /** read parent PK spec (type + unsigned) */
    private function pkSpec(string $table): ?array
    {
        $row = DB::selectOne("
            SELECT DATA_TYPE, COLUMN_TYPE
            FROM information_schema.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE()
              AND TABLE_NAME = ?
              AND COLUMN_NAME = 'id'
            LIMIT 1
        ", [$table]);

        if (!$row) return null;
        $type = strtolower($row->DATA_TYPE); // int | bigint | ...
        $unsigned = str_contains(strtolower($row->COLUMN_TYPE), 'unsigned');
        return ['type'=>$type, 'unsigned'=>$unsigned];
    }

    private function sqlType(array $spec, bool $nullable): string
    {
        $t = match ($spec['type']) {
            'int'    => 'INT',
            'bigint' => 'BIGINT',
            default  => strtoupper($spec['type']),
        };
        if ($spec['unsigned']) $t .= ' UNSIGNED';
        $t .= $nullable ? ' NULL' : ' NOT NULL';
        return $t;
    }

    public function up(): void
    {
        if (!Schema::hasTable($this->table)) return;

        // Child -> parent mapping (edit names if your parents differ)
        $map = [
            'bag_type_id' => 'bag_types',
            'company_id'  => 'companies',
            'supplier_id' => 'suppliers',
            'employee_id' => 'employees',
            'created_by'  => 'users',
            'updated_by'  => 'users',
        ];

        // 1) Align child column types to parent PK types (prevents errno:150)
        foreach ($map as $col => $parent) {
            if (!Schema::hasTable($parent)) continue;
            if (!Schema::hasColumn($this->table, $col)) continue;

            $spec = $this->pkSpec($parent);
            if (!$spec) continue;

            // created_by is NOT NULL in dump; others are NULLable
            $nullable = !in_array($col, ['created_by'], true);
            $sqlType  = $this->sqlType($spec, $nullable);
            DB::statement("ALTER TABLE `{$this->table}` MODIFY `{$col}` {$sqlType}");
        }

        // 2) Add missing FKs (skip gracefully if parent missing)
        Schema::table($this->table, function (Blueprint $t) use ($map) {
            foreach ($map as $col => $parent) {
                if (!Schema::hasTable($parent)) continue;
                if ($this->hasForeignOn($col))   continue;

                $fk = "rm_bag_ledgers_{$col}_foreign";
                $t->foreign($col, $fk)
                  ->references('id')->on($parent)
                  ->cascadeOnUpdate()
                  // created_by = NOT NULL -> restrict; others NULLable -> set null
                  ->{in_array($col, ['created_by'], true) ? 'restrictOnDelete' : 'nullOnDelete'}();
            }
        });
    }

    public function down(): void
    {
        if (!Schema::hasTable($this->table)) return;

        Schema::table($this->table, function (Blueprint $t) {
            foreach ([
                'rm_bag_ledgers_bag_type_id_foreign',
                'rm_bag_ledgers_company_id_foreign',
                'rm_bag_ledgers_supplier_id_foreign',
                'rm_bag_ledgers_employee_id_foreign',
                'rm_bag_ledgers_created_by_foreign',
                'rm_bag_ledgers_updated_by_foreign',
            ] as $fk) {
                try { $t->dropForeign($fk); } catch (\Throwable $e) {}
            }
        });
    }
};

