<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    private string $table = 'notifications';

    /** কোনো কলামে আগেই FK আছে কি না (constraint নাম যাই হোক) */
    private function hasForeignOn(string $column): bool
    {
        $sql = <<<SQL
SELECT 1
FROM information_schema.KEY_COLUMN_USAGE
WHERE TABLE_SCHEMA = DATABASE()
  AND TABLE_NAME   = ?
  AND COLUMN_NAME  = ?
  AND REFERENCED_TABLE_NAME IS NOT NULL
LIMIT 1
SQL;
        return !empty(DB::select($sql, [$this->table, $column]));
    }

    public function up(): void
    {
        if (!Schema::hasTable($this->table)) return;

        // কোন parent টেবিলগুলো রয়েছে—শুধু সেগুলোতেই FK বসাবো
        $parents = [
            'company_id'     => 'companies',
            'department_id'  => 'departments',
            'designation_id' => 'designations',
            'order_type_id'  => 'order_types',
            'buyer_id'       => 'buyers',
            'employee_id'    => 'employees',
            // 'source_id' => ???  // এটি সাধারণত polymorphic হয়ে থাকে (type+id); তাই FK না দেওয়াই ভালো
        ];

        $needWork = false;
        foreach ($parents as $col => $parent) {
            if (Schema::hasTable($parent) && !$this->hasForeignOn($col)) {
                $needWork = true; break;
            }
        }
        if (!$needWork) return;

        Schema::table($this->table, function (Blueprint $table) use ($parents) {
            foreach ($parents as $col => $parent) {
                if (Schema::hasTable($parent)) {
                    // আগেই FK না থাকলেই add করবো
                    // nullable কলাম হলে ->nullOnDelete(), না হলে ->restrictOnDelete()
                    $fkName = "{$this->table}_{$col}_foreign";

                    // nullable কিনা বুঝে ডিলিট রুল সেট করা
                    $nullableCols = ['company_id','department_id','designation_id','order_type_id','buyer_id','employee_id'];
                    $deleteRule = in_array($col, $nullableCols, true) ? 'null' : 'restrict';

                    // আগেই থাকলে exception এড়াতে try/catch ব্যবহার করতে পারেন, তবে আমরা existence আগে চেক করেছি
                    if (!DB::selectOne("
                        SELECT 1
                        FROM information_schema.KEY_COLUMN_USAGE
                        WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? AND COLUMN_NAME = ?
                          AND REFERENCED_TABLE_NAME IS NOT NULL
                        LIMIT 1
                    ", [$this->table, $col])) {
                        $table->foreign($col, $fkName)
                              ->references('id')->on($parent)
                              ->cascadeOnUpdate();

                        // delete rule set
                        if ($deleteRule === 'null') {
                            $table->dropForeign($fkName); // Need to rebuild with proper rule in some drivers
                            $table->foreign($col, $fkName)
                                  ->references('id')->on($parent)
                                  ->cascadeOnUpdate()
                                  ->nullOnDelete();
                        } else {
                            $table->dropForeign($fkName);
                            $table->foreign($col, $fkName)
                                  ->references('id')->on($parent)
                                  ->cascadeOnUpdate()
                                  ->restrictOnDelete();
                        }
                    }
                }
            }
        });
    }

    public function down(): void
    {
        if (!Schema::hasTable($this->table)) return;

        Schema::table($this->table, function (Blueprint $table) {
            foreach ([
                'notifications_company_id_foreign',
                'notifications_department_id_foreign',
                'notifications_designation_id_foreign',
                'notifications_order_type_id_foreign',
                'notifications_buyer_id_foreign',
                'notifications_employee_id_foreign',
            ] as $fk) {
                try { $table->dropForeign($fk); } catch (\Throwable $e) {}
            }
        });
    }
};
