<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    private string $table = 'item_units';

    // group_id যে টেবিলকে রেফার করবে (প্রয়োজনে 'groups' যোগ করুন)
    private array $groupParents = ['item_groups'];

    private function pickExisting(array $names): ?string
    {
        foreach ($names as $n) if (Schema::hasTable($n)) return $n;
        return null;
    }

    private function hasForeignOn(string $table, string $column): bool
    {
        $sql = <<<SQL
SELECT 1
FROM information_schema.KEY_COLUMN_USAGE
WHERE TABLE_SCHEMA = DATABASE()
  AND TABLE_NAME   = ?
  AND COLUMN_NAME  = ?
  AND REFERENCED_TABLE_NAME IS NOT NULL
LIMIT 1
SQL;
        return !empty(DB::select($sql, [$table, $column]));
    }

    /** parent PK spec: ['type' => 'int'|'bigint', 'unsigned'=>bool] */
    private function pkSpec(string $table): ?array
    {
        $row = DB::selectOne("
            SELECT DATA_TYPE, COLUMN_TYPE
            FROM information_schema.COLUMNS
            WHERE TABLE_SCHEMA = DATABASE()
              AND TABLE_NAME = ?
              AND COLUMN_NAME = 'id'
            LIMIT 1
        ", [$table]);

        if (!$row) return null;
        $type = strtolower($row->DATA_TYPE);
        $unsigned = str_contains(strtolower($row->COLUMN_TYPE), 'unsigned');
        return ['type'=>$type, 'unsigned'=>$unsigned];
    }

    private function sqlTypeFromSpec(array $spec, bool $nullable=false): string
    {
        $t = match ($spec['type']) {
            'int' => 'INT',
            'bigint' => 'BIGINT',
            default => strtoupper($spec['type']),
        };
        if ($spec['unsigned']) $t .= ' UNSIGNED';
        $t .= $nullable ? ' NULL' : ' NOT NULL';
        return $t;
    }

    public function up(): void
    {
        if (!Schema::hasTable($this->table)) return;

        // Detect parents (যেটা আছে সেটাই নেব)
        $groupParent   = $this->pickExisting($this->groupParents);
        $usersParent   = Schema::hasTable('users') ? 'users' : null;

        // FK already আছে কি না (by column)
        $needGroupFk   = $groupParent && !$this->hasForeignOn($this->table, 'group_id');
        $needCreatedFk = $usersParent && !$this->hasForeignOn($this->table, 'created_by');
        $needUpdatedFk = $usersParent && !$this->hasForeignOn($this->table, 'updated_by');

        // Types align: child columns -> parent PK exact type (INT/BIGINT + UNSIGNED)
        // (Doctrine লাগবে না; raw ALTER দিয়ে করব)
        if ($needGroupFk) {
            $spec = $this->pkSpec($groupParent);
            if ($spec) {
                DB::statement("ALTER TABLE `{$this->table}` MODIFY `group_id` {$this->sqlTypeFromSpec($spec, false)}");
            }
        }
        if ($needCreatedFk) {
            $spec = $this->pkSpec($usersParent);
            if ($spec) {
                DB::statement("ALTER TABLE `{$this->table}` MODIFY `created_by` {$this->sqlTypeFromSpec($spec, false)}");
            }
        }
        if ($needUpdatedFk) {
            $spec = $this->pkSpec($usersParent);
            if ($spec) {
                DB::statement("ALTER TABLE `{$this->table}` MODIFY `updated_by` {$this->sqlTypeFromSpec($spec, false)}");
            }
        }

        // Now add the available FKs (missing ones only)
        if ($needGroupFk || $needCreatedFk || $needUpdatedFk) {
            Schema::table($this->table, function (Blueprint $t) use ($groupParent, $usersParent, $needGroupFk, $needCreatedFk, $needUpdatedFk) {
                if ($needGroupFk && $groupParent) {
                    $t->foreign('group_id', 'item_units_group_id_foreign')
                      ->references('id')->on($groupParent)
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();
                }
                if ($needCreatedFk && $usersParent) {
                    $t->foreign('created_by', 'item_units_created_by_foreign')
                      ->references('id')->on($usersParent)
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();
                }
                if ($needUpdatedFk && $usersParent) {
                    $t->foreign('updated_by', 'item_units_updated_by_foreign')
                      ->references('id')->on($usersParent)
                      ->cascadeOnUpdate()
                      ->restrictOnDelete();
                }
            });
        }
        // Parent না থাকলে কিছুই করবে না—মাইগ্রেশন পাস করবে।
    }

    public function down(): void
    {
        if (!Schema::hasTable($this->table)) return;

        Schema::table($this->table, function (Blueprint $t) {
            foreach ([
                'item_units_group_id_foreign',
                'item_units_created_by_foreign',
                'item_units_updated_by_foreign',
            ] as $fk) {
                try { $t->dropForeign($fk); } catch (\Throwable $e) {}
            }
        });
    }
};

