<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    // Primary table (fallback to candidates if you renamed it)
    private string $preferredTable = 'issue_raw_material_details';

    // Fallback names if your table might differ
    private array $candidates = [
        'issue_raw_material_details',
        'issue_raw_material_detail',
        'issue_raw_materials_details',
        'issue_raw_materials_detail',
    ];

    // Candidate parent tables for material_id (put your real one first if you know it)
    private array $materialParents = ['materials', 'raw_materials', 'products'];

    private function pickExistingTable(array $names): ?string
    {
        foreach ($names as $n) {
            if (Schema::hasTable($n)) return $n;
        }
        return null;
    }

    /** Check if a column already has a FK (any name) */
    private function hasForeignOnColumn(string $table, string $column): bool
    {
        $sql = <<<SQL
SELECT 1
FROM information_schema.KEY_COLUMN_USAGE
WHERE TABLE_SCHEMA = DATABASE()
  AND TABLE_NAME   = ?
  AND COLUMN_NAME  = ?
  AND REFERENCED_TABLE_NAME IS NOT NULL
LIMIT 1
SQL;
        return !empty(DB::select($sql, [$table, $column]));
    }

    public function up(): void
    {
        // Resolve child table (do not throw; just no-op if not found)
        $tbl = Schema::hasTable($this->preferredTable)
            ? $this->preferredTable
            : $this->pickExistingTable($this->candidates);

        if (!$tbl) {
            // Table not present yet — nothing to do
            return;
        }

        // Detect which material parent exists (if any)
        $materialParent = $this->pickExistingTable($this->materialParents);
        $hasWarehouses  = Schema::hasTable('warehouses');

        // Determine which FKs are actually needed (and possible)
        $needMaterialFk  = $materialParent && !$this->hasForeignOnColumn($tbl, 'material_id');
        $needWarehouseFk = $hasWarehouses  && !$this->hasForeignOnColumn($tbl, 'warehouse_id');

        // If neither parent exists, simply do nothing (no exceptions)
        if (!$needMaterialFk && !$needWarehouseFk) {
            return;
        }

        Schema::table($tbl, function (Blueprint $t) use ($tbl, $materialParent, $needMaterialFk, $needWarehouseFk) {
            // Use Laravel’s default-style names for consistency
            $materialFkName  = "{$tbl}_material_id_foreign";
            $warehouseFkName = "{$tbl}_warehouse_id_foreign";

            if ($needMaterialFk && $materialParent) {
                $t->foreign('material_id', $materialFkName)
                    ->references('id')->on($materialParent)
                    ->cascadeOnUpdate()
                    ->restrictOnDelete(); // change if you prefer ->nullOnDelete() / ->cascadeOnDelete()
            }

            if ($needWarehouseFk) {
                $t->foreign('warehouse_id', $warehouseFkName)
                    ->references('id')->on('warehouses')
                    ->cascadeOnUpdate()
                    ->nullOnDelete(); // nullable FK per your schema
            }
        });
    }

    public function down(): void
    {
        $tbl = Schema::hasTable($this->preferredTable)
            ? $this->preferredTable
            : $this->pickExistingTable($this->candidates);

        if (!$tbl) return;

        Schema::table($tbl, function (Blueprint $t) use ($tbl) {
            foreach ([
                "{$tbl}_material_id_foreign",
                "{$tbl}_warehouse_id_foreign",
            ] as $fk) {
                try { $t->dropForeign($fk); } catch (\Throwable $e) {}
            }
        });
    }
};




