<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // Detect possible table names
        $suppliersTable = Schema::hasTable('acc_suppliers') ? 'acc_suppliers'
                        : (Schema::hasTable('suppliers') ? 'suppliers' : null);

        $employeesTable = Schema::hasTable('employees') ? 'employees'
                        : (Schema::hasTable('acc_employees') ? 'acc_employees' : null);

        $warehouseTable = Schema::hasTable('warehouses') ? 'warehouses'
                        : (Schema::hasTable('acc_warehouses') ? 'acc_warehouses' : null);

        Schema::table('bag_purchases', function (Blueprint $table) use ($suppliersTable, $employeesTable, $warehouseTable) {
            // company (nullable → SET NULL)
            if (Schema::hasTable('companies')) {
                $table->foreign('company_id', 'bag_purchases_company_fk')
                      ->references('id')->on('companies')
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // warehouse (nullable → SET NULL)
            if ($warehouseTable) {
                $table->foreign('warehouse_id', 'bag_purchases_wh_fk')
                      ->references('id')->on($warehouseTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // supplier (nullable → SET NULL) — your dump shows index but FK name varies; we add one
            if ($suppliersTable) {
                $table->foreign('supplier_id', 'bag_purchases_supplier_fk')
                      ->references('id')->on($suppliersTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // employee (nullable → SET NULL)
            if ($employeesTable) {
                $table->foreign('employee_id', 'bag_purchases_employee_fk')
                      ->references('id')->on($employeesTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // users (created_by/updated_by are nullable in the dump)
            if (Schema::hasTable('users')) {
                $table->foreign('created_by', 'bag_purchases_created_by_fk')
                      ->references('id')->on('users')
                      ->onUpdate('cascade')
                      ->onDelete('set null');

                $table->foreign('updated_by', 'bag_purchases_updated_by_fk')
                      ->references('id')->on('users')
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }
        });
    }

    public function down(): void
    {
        Schema::table('bag_purchases', function (Blueprint $table) {
            foreach ([
                'bag_purchases_company_fk',
                'bag_purchases_wh_fk',
                'bag_purchases_supplier_fk',
                'bag_purchases_employee_fk',
                'bag_purchases_created_by_fk',
                'bag_purchases_updated_by_fk',
            ] as $fk) {
                try { $table->dropForeign($fk); } catch (\Throwable $e) {}
            }
        });
    }
};
