<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // Detect product master table
        $productTable = null;
        if (Schema::hasTable('products')) {
            $productTable = 'products';
        } elseif (Schema::hasTable('finish_products')) {
            $productTable = 'finish_products';
        } elseif (Schema::hasTable('raw_materials')) {
            $productTable = 'raw_materials';
        }

        // Detect bag master table
        $bagTable = null;
        if (Schema::hasTable('finish_bags')) {
            $bagTable = 'finish_bags';
        } elseif (Schema::hasTable('bags')) {
            $bagTable = 'bags';
        } elseif (Schema::hasTable('raw_bags')) {
            $bagTable = 'raw_bags';
        } elseif (Schema::hasTable('raw_bag')) { // in case singular is used
            $bagTable = 'raw_bag';
        }

        // Detect warehouse table name
        $warehouseTable = Schema::hasTable('warehouses') ? 'warehouses'
                         : (Schema::hasTable('acc_warehouses') ? 'acc_warehouses' : null);

        Schema::table('bag_exchanges', function (Blueprint $table) use ($productTable, $bagTable, $warehouseTable) {
            // company (nullable → SET NULL)
            if (Schema::hasTable('companies')) {
                $table->foreign('company_id', 'be_company_fk')
                      ->references('id')->on('companies')
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // warehouses (nullable → SET NULL)
            if ($warehouseTable) {
                $table->foreign('warehouse_id', 'be_wh_fk')
                      ->references('id')->on($warehouseTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');

                $table->foreign('exchange_warehouse_id', 'be_ex_wh_fk')
                      ->references('id')->on($warehouseTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');

                $table->foreign('bag_warehouse_id', 'be_bag_wh_fk')
                      ->references('id')->on($warehouseTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');

                $table->foreign('bag_exchange_warehouse_id', 'be_ex_bag_wh_fk')
                      ->references('id')->on($warehouseTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // products (nullable → RESTRICT)
            if ($productTable) {
                $table->foreign('product_id', 'be_product_fk')
                      ->references('id')->on($productTable)
                      ->onUpdate('cascade')
                      ->onDelete('restrict');

                $table->foreign('exchange_product_id', 'be_ex_product_fk')
                      ->references('id')->on($productTable)
                      ->onUpdate('cascade')
                      ->onDelete('restrict');
            }

            // bags (nullable → RESTRICT)
            if ($bagTable) {
                $table->foreign('bag_id', 'be_bag_fk')
                      ->references('id')->on($bagTable)
                      ->onUpdate('cascade')
                      ->onDelete('restrict');

                $table->foreign('exchange_bag_id', 'be_ex_bag_fk')
                      ->references('id')->on($bagTable)
                      ->onUpdate('cascade')
                      ->onDelete('restrict');
            }

            // users (nullable in schema → SET NULL)
            if (Schema::hasTable('users')) {
                $table->foreign('created_by', 'be_created_by_fk')
                      ->references('id')->on('users')
                      ->onUpdate('cascade')
                      ->onDelete('set null');

                $table->foreign('updated_by', 'be_updated_by_fk')
                      ->references('id')->on('users')
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }
        });
    }

    public function down(): void
    {
        Schema::table('bag_exchanges', function (Blueprint $table) {
            // Guarded drops (ignore if not present)
            foreach ([
                'be_company_fk', 'be_wh_fk', 'be_ex_wh_fk', 'be_bag_wh_fk', 'be_ex_bag_wh_fk',
                'be_product_fk', 'be_ex_product_fk', 'be_bag_fk', 'be_ex_bag_fk',
                'be_created_by_fk', 'be_updated_by_fk'
            ] as $fk) {
                try { $table->dropForeign($fk); } catch (\Throwable $e) {}
            }
        });
    }
};
