<?php

use Illuminate\Database\Migrations\Migration;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Support\Facades\Schema;

return new class extends Migration {
    public function up(): void
    {
        // Detect product master table
        $productTable = null;
        if (Schema::hasTable('products')) {
            $productTable = 'products';
        } elseif (Schema::hasTable('finish_products')) {
            $productTable = 'finish_products';
        } elseif (Schema::hasTable('raw_materials')) {
            $productTable = 'raw_materials';
        }

        // Detect branch/warehouse tables (adjust if different in your app)
        $branchTable    = Schema::hasTable('branches')   ? 'branches'
                         : (Schema::hasTable('acc_branches')   ? 'acc_branches'   : null);
        $warehouseTable = Schema::hasTable('warehouses') ? 'warehouses'
                         : (Schema::hasTable('acc_warehouses') ? 'acc_warehouses' : null);

        Schema::table('acc_stock_summaries', function (Blueprint $table) use ($productTable, $branchTable, $warehouseTable) {
            // company (required → RESTRICT)
            if (Schema::hasTable('companies')) {
                $table->foreign('company_id', 'ass_company_fk')
                      ->references('id')->on('companies')
                      ->onUpdate('cascade')
                      ->onDelete('restrict');
            }

            // product (required → RESTRICT)
            if ($productTable) {
                $table->foreign('product_id', 'ass_product_fk')
                      ->references('id')->on($productTable)
                      ->onUpdate('cascade')
                      ->onDelete('restrict');
            }

            // branch (nullable → SET NULL)
            if ($branchTable) {
                $table->foreign('branch_id', 'ass_branch_fk')
                      ->references('id')->on($branchTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // warehouse (nullable → SET NULL)
            if ($warehouseTable) {
                $table->foreign('warehouse_id', 'ass_wh_fk')
                      ->references('id')->on($warehouseTable)
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }

            // users (created_by required, updated_by nullable)
            if (Schema::hasTable('users')) {
                $table->foreign('created_by', 'ass_created_by_fk')
                      ->references('id')->on('users')
                      ->onUpdate('cascade')
                      ->onDelete('restrict');

                $table->foreign('updated_by', 'ass_updated_by_fk')
                      ->references('id')->on('users')
                      ->onUpdate('cascade')
                      ->onDelete('set null');
            }
        });
    }

    public function down(): void
    {
        Schema::table('acc_stock_summaries', function (Blueprint $table) {
            // Drop by names; guard if some weren’t created
            try { $table->dropForeign('ass_company_fk'); } catch (\Throwable $e) {}
            try { $table->dropForeign('ass_product_fk'); } catch (\Throwable $e) {}
            try { $table->dropForeign('ass_branch_fk'); } catch (\Throwable $e) {}
            try { $table->dropForeign('ass_wh_fk'); } catch (\Throwable $e) {}
            try { $table->dropForeign('ass_created_by_fk'); } catch (\Throwable $e) {}
            try { $table->dropForeign('ass_updated_by_fk'); } catch (\Throwable $e) {}
        });
    }
};

