<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class RawMaterial extends Model
{
    use SoftDeletes;

    // Table name (because it's not the plural of the model)
    protected $table = 'rm_products';

    // Fields you can mass-assign
    protected $fillable = [
        'name',
        'category_id',
        'warehouse_id',
        'supplier_id',
        'opening_qty',
        'alert_qty',
        'vat',
        'price',
        'status',
        'type',          // keep 'raw' as default/marker
        'image_path',
        'description',
        'created_by',
        'updated_by',
    ];

    // Cast numerics properly for validation + calculations
    protected $casts = [
        'opening_qty' => 'decimal:4',
        'alert_qty'   => 'decimal:4',
        'vat'         => 'decimal:4',
        'price'       => 'decimal:4',
        'status'      => 'integer',
    ];

    /* ---------------- Relations ---------------- */
    public function category()  { return $this->belongsTo(Category::class); }
    public function warehouse() { return $this->belongsTo(Warehouse::class); }
    public function supplier()  { return $this->belongsTo(Supplier::class); }

    public function createdBy() { return $this->belongsTo(User::class, 'created_by'); }
    public function updatedBy() { return $this->belongsTo(User::class, 'updated_by'); }

    /* ---------------- Accessors ---------------- */
    // Convenient URL for the image (returns null if not set)
    public function getImageUrlAttribute(): ?string
    {
        return $this->image_path ? asset($this->image_path) : null;
    }

    /* ---------------- Model Events ---------------- */
    protected static function booted(): void
    {
        static::creating(function (self $m) {
            // Ensure type is always 'raw'
            $m->type       = $m->type ?: 'raw';
            $m->created_by = auth()->id() ?? $m->created_by;
            $m->updated_by = auth()->id() ?? $m->updated_by;
        });

        static::updating(function (self $m) {
            $m->updated_by = auth()->id() ?? $m->updated_by;
        });
    }
}
