<?php

namespace App\Models;

use App\Models\User;
use App\Models\Category;   // rm_categories
use App\Models\Supplier;   // rm_suppliers
use App\Models\Warehouse;  // warehouses
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Builder;

class FinishProduct extends Model
{
    use HasFactory, SoftDeletes;

    /**
     * This model shares the rm_products table with Raw materials.
     * We add a global scope so every query enforces type = 2 (finish).
     */
    protected $table = 'rm_products';

    /** Mass-assignable columns */
    protected $fillable = [
        'name',
        'category_id',
        'unit_id',
        'supplier_id',
        'small_unit_id',
        'big_unit_id',
        'warehouse_id',
        'bag_warehouse_id',

        'purchase_price',
        'sale_price',
        'opening_quantity',
        'alert_quantity',
        'vat',
        'conversion_to_kg',

        'description',
        'image',                 // stores 'storage/...' path if uploaded
        'status',                // 0/1

        'bag_opening_quantity',
        'stock_type',
        'product_price',
        'bag_price',

        'created_by',
        'updated_by',

        // 'type' is intentionally omitted from $fillable to avoid accidental change
    ];

    /** Attribute casting */
    protected $casts = [
        'purchase_price'       => 'decimal:2',
        'sale_price'           => 'decimal:2',
        'opening_quantity'     => 'decimal:2',
        'alert_quantity'       => 'decimal:2',
        'vat'                  => 'decimal:2',
        'conversion_to_kg'     => 'decimal:2',
        'bag_opening_quantity' => 'decimal:2',
        'product_price'        => 'decimal:2',
        'bag_price'            => 'decimal:2',
        'status'               => 'integer',
    ];

    /** Default attributes (ensure type is finish) */
    protected $attributes = [
        'status' => 1,
    ];

    /* -------------------------------------------------------------
     | Global scope & model events
     |-------------------------------------------------------------*/
    protected static function booted(): void
    {
        // Always limit to finish products (type = 2)
        static::addGlobalScope('finishOnly', function (Builder $q) {
            $q->where('type', 2);
        });

        // On create: force type=2 and set created_by/updated_by if present
        static::creating(function (self $m) {
            $m->type       = 2; // finish
            $uid           = auth()->id();
            $m->created_by = $m->created_by ?? $uid;
            $m->updated_by = $m->updated_by ?? $uid;
        });

        // On update: keep updated_by fresh
        static::updating(function (self $m) {
            $m->updated_by = auth()->id() ?? $m->updated_by;
        });
    }

    /* -------------------------------------------------------------
     | Relationships
     |-------------------------------------------------------------*/
    public function category()
    {
        // table: rm_categories
        return $this->belongsTo(Category::class, 'category_id');
    }

    public function supplier()
    {
        // table: rm_suppliers
        return $this->belongsTo(Supplier::class, 'supplier_id');
    }

    public function warehouse()
    {
        return $this->belongsTo(Warehouse::class, 'warehouse_id');
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by')->withDefault();
    }

    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by')->withDefault();
    }

    /* -------------------------------------------------------------
     | Accessors / Helpers
     |-------------------------------------------------------------*/

    /**
     * Public URL of the stored image (if any)
     * Usage: $product->image_url
     */
    public function getImageUrlAttribute(): ?string
    {
        $path = $this->image; // expected like 'storage/products/xyz.jpg'
        if (!$path) return null;
        // If it already looks public, return asset() path
        return asset($path);
    }

    /**
     * Nicely formatted created date for listings
     * Usage: $product->created_on
     */
    public function getCreatedOnAttribute(): ?string
    {
        return optional($this->created_at)->format('d/m/Y');
    }

    /**
     * Status label (Active/Inactive)
     * Usage: $product->status_label
     */
    public function getStatusLabelAttribute(): string
    {
        return ((int)$this->status) === 1 ? 'Active' : 'Inactive';
    }

    /* -------------------------------------------------------------
     | Local scopes (optional extras)
     |-------------------------------------------------------------*/
    public function scopeActive(Builder $q): Builder
    {
        return $q->where('status', 1);
    }

    public function scopeSearch(Builder $q, ?string $term): Builder
    {
        $term = trim((string)$term);
        if ($term === '') return $q;

        return $q->where(function ($w) use ($term) {
            $w->where('name', 'like', "%{$term}%")
              ->orWhereHas('category', fn ($c) => $c->where('name', 'like', "%{$term}%"));
        });
    }
}
