<?php

namespace App\Models;

use App\Models\User;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Factories\HasFactory;

class Category extends Model
{
    use HasFactory;

    /** Table name */
    protected $table = 'rm_categories';

    /** Mass assignable */
    protected $fillable = [
        'name',
        'code',
        'status',     // stored as '1' / '0'
        'created_by',
        'updated_by',
    ];

    /** Casts */
    protected $casts = [
        // keep as string so it stays '1' / '0'
        'status' => 'string',
    ];

    /* -------------------------------------------------------------------------
     | Relationships
     |-------------------------------------------------------------------------*/
    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function updatedBy()
    {
        return $this->belongsTo(User::class, 'updated_by');
    }

    /* -------------------------------------------------------------------------
     | Accessors (created_by_name / updated_by_name)
     |  - Works whether or not User uses SoftDeletes
     |-------------------------------------------------------------------------*/
    public function getCreatedByNameAttribute(): string
    {
        // If relation is eager-loaded, use it; otherwise fetch safely.
        $user = $this->relationLoaded('createdBy')
            ? $this->createdBy
            : $this->findUserSafely($this->created_by);

        return $user->name ?? '—';
    }

    public function getUpdatedByNameAttribute(): string
    {
        $user = $this->relationLoaded('updatedBy')
            ? $this->updatedBy
            : $this->findUserSafely($this->updated_by);

        return $user->name ?? '—';
    }

    /**
     * Fetch a User by id.
     * If the User model uses SoftDeletes, include trashed records; otherwise normal find().
     */
    protected function findUserSafely(?int $id): ?User
    {
        if (!$id) {
            return null;
        }

        // Detect if the User model uses SoftDeletes without importing it.
        $usesSoftDeletes = in_array(
            \Illuminate\Database\Eloquent\SoftDeletes::class,
            class_uses_recursive(User::class),
            true
        );

        $query = User::query();
        if ($usesSoftDeletes && method_exists($query, 'withTrashed')) {
            $query->withTrashed();
        }

        return $query->find($id);
    }

    /* -------------------------------------------------------------------------
     | Model events (auto fill created_by / updated_by)
     |-------------------------------------------------------------------------*/
    protected static function booted(): void
    {
        static::creating(function (self $model) {
            if (is_null($model->created_by)) {
                $model->created_by = auth()->id() ?? 1; // ensure a valid fallback user exists
            }
        });

        static::updating(function (self $model) {
            $model->updated_by = auth()->id() ?? $model->updated_by;
        });
    }
}


