<?php

namespace App\Http\Controllers;

use App\Models\WarehouseType;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class WarehouseTypeController extends Controller
{
    /**
     * List page (search + paginate)
     */
    public function index(Request $request)
    {
        $q = trim((string)$request->input('q'));

        $types = WarehouseType::query()
            ->when($q !== '', function ($query) use ($q) {
                $query->where('name', 'like', "%{$q}%")
                      ->orWhere('status', 'like', "%{$q}%");
            })
            ->orderBy('id')
            ->paginate(20)
            ->appends(['q' => $q]);

        return view('products.warehouse-types.index', compact('types'));
    }

    /**
     * Create form
     */
    public function create()
    {
        return view('products.warehouse-types.create');
    }

    /**
     * Store
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'   => ['required', 'string', 'max:255', 'unique:rm_warehouse_types,name'],
            'status' => ['nullable', Rule::in(['0','1',0,1])],
        ]);

        WarehouseType::create([
            'name'       => $data['name'],
            'status'     => isset($data['status']) ? (string)$data['status'] : '1',
            'created_by' => auth()->id(),
        ]);

        return redirect()
            ->route('products.warehouse-types.index')
            ->with('success', 'Warehouse type created successfully.');
    }

    /**
     * Edit form
     */
    public function edit(WarehouseType $warehouseType)
    {
        return view('products.warehouse-types.edit', ['type' => $warehouseType]);
    }

    /**
     * Update
     */
    public function update(Request $request, WarehouseType $warehouseType)
    {
        $data = $request->validate([
            'name'   => [
                'required','string','max:255',
                Rule::unique('rm_warehouse_types','name')->ignore($warehouseType->id),
            ],
            'status' => ['nullable', Rule::in(['0','1',0,1])],
        ]);

        $warehouseType->update([
            'name'       => $data['name'],
            'status'     => isset($data['status']) ? (string)$data['status'] : $warehouseType->status,
            'updated_by' => auth()->id(),
        ]);

        return redirect()
            ->route('products.warehouse-types.index')
            ->with('success', 'Warehouse type updated successfully.');
    }

    /**
     * Destroy
     */
    public function destroy(WarehouseType $warehouseType)
    {
        try {
            $warehouseType->delete();
            return back()->with('success', 'Warehouse type removed.');
        } catch (\Throwable $e) {
            return back()->with('error', 'Cannot delete this item (it may be in use).');
        }
    }
}
