<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\View;

class UserController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth'); // সব অ্যাকশনেই auth দরকার
    }

    /* -----------------------------------------------------------
     | Utility: pick existing view path without touching blades
     | Tries 'access.users.{name}' first, then 'users.{name}'
     ----------------------------------------------------------- */
    protected function viewPath(string $name): string
    {
        $a = "access.users.$name";
        $b = "users.$name";
        return View::exists($a) ? $a : $b;
    }

    /* -----------------------------------------------------------
     | GET /users  -> users.index
     | Optional search: ?q=
     ----------------------------------------------------------- */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $users = User::query()
            ->when(Auth::user()?->company_id, function ($qr) {
                // নিজের কোম্পানির ইউজার—থাকলে ফিল্টার (আপনার চাহিদা অনুযায়ী)
                $qr->where('company_id', Auth::user()->company_id);
            })
            ->when($q !== '', function ($qr) use ($q) {
                $qr->where(function ($w) use ($q) {
                    $w->where('name', 'like', "%{$q}%")
                      ->orWhere('email', 'like', "%{$q}%");
                });
            })
            ->orderByDesc('id')
            ->paginate(20);

        return view($this->viewPath('index'), compact('users', 'q'));
    }

    /* -----------------------------------------------------------
     | GET /users/create  -> users.create
     ----------------------------------------------------------- */
    public function create()
    {
        return view($this->viewPath('create'));
    }

    /* -----------------------------------------------------------
     | POST /users -> users.store
     ----------------------------------------------------------- */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'       => ['required', 'string', 'max:255'],
            'email'      => ['required', 'email', 'max:255', 'unique:users,email'],
            'password'   => ['required', 'min:8', 'confirmed'],
            'company_id' => ['nullable', 'integer'], // আপনার টেবিল/রুল থাকলে exists দিন
            'branch_id'  => ['nullable', 'integer'],
            'status'     => ['nullable', 'boolean'],
        ]);

        $user = new User();
        $user->name       = $validated['name'];
        $user->email      = $validated['email'];
        // আপনার User মডেলে password mutator আছে—plain text সেট করলেই hash হবে
        $user->password   = $validated['password'];

        // company/branch fallback
        $user->company_id = $validated['company_id']
            ?? (Auth::user()->company_id ?? 1);
        $user->branch_id  = $validated['branch_id'] ?? null;

        $user->status     = array_key_exists('status', $validated) ? (bool)$validated['status'] : true;
        $user->remember_token = Str::random(10);

        $user->save();

        return redirect()->route('users.index')->with('status', 'User created successfully.');
    }

    /* -----------------------------------------------------------
     | GET /users/{user} -> users.show
     ----------------------------------------------------------- */
    public function show(User $user)
    {
        return view($this->viewPath('show'), compact('user'));
    }

    /* -----------------------------------------------------------
     | GET /users/{user}/edit -> users.edit
     ----------------------------------------------------------- */
    public function edit(User $user)
    {
        return view($this->viewPath('edit'), compact('user'));
    }

    /* -----------------------------------------------------------
     | PUT /users/{user} -> users.update
     ----------------------------------------------------------- */
    public function update(Request $request, User $user)
    {
        $validated = $request->validate([
            'name'     => ['required', 'string', 'max:255'],
            'email'    => [
                'required', 'email', 'max:255',
                Rule::unique('users', 'email')->ignore($user->id),
            ],
            'password' => ['nullable', 'min:8', 'confirmed'],
            'company_id' => ['nullable', 'integer'],
            'branch_id'  => ['nullable', 'integer'],
            'status'     => ['nullable', 'boolean'],
        ]);

        $user->name  = $validated['name'];
        $user->email = $validated['email'];

        if (!empty($validated['password'])) {
            $user->password = $validated['password']; // mutator hash করবে
        }

        // ঐচ্ছিক ফিল্ড
        if (array_key_exists('company_id', $validated)) {
            $user->company_id = $validated['company_id'];
        }
        if (array_key_exists('branch_id', $validated)) {
            $user->branch_id = $validated['branch_id'];
        }
        if (array_key_exists('status', $validated)) {
            // নিজেকে নিষ্ক্রিয় করতে না দেওয়া (ঐচ্ছিক সেফটি)
            if ($user->id === Auth::id() && !$validated['status']) {
                return back()->withErrors(['status' => 'You cannot disable your own account.']);
            }
            $user->status = (bool)$validated['status'];
        }

        $user->save();

        return redirect()
            ->route('users.edit', $user)
            ->with('status', 'User updated successfully.');
    }

    /* -----------------------------------------------------------
     | DELETE /users/{user} -> users.destroy
     ----------------------------------------------------------- */
    public function destroy(User $user)
    {
        if ($user->id === Auth::id()) {
            return back()->withErrors(['delete' => 'You cannot delete your own account.']);
        }

        $user->delete();
        return redirect()->route('users.index')->with('status', 'User deleted.');
    }

    /* -----------------------------------------------------------
     | PATCH /users/{user}/toggle -> users.toggle
     | Enable/Disable user (status flip)
     ----------------------------------------------------------- */
    public function toggle(User $user)
    {
        if ($user->id === Auth::id()) {
            return back()->withErrors(['toggle' => 'You cannot disable your own account.']);
        }

        $user->status = !$user->status;
        $user->save();

        return back()->with('status', $user->status ? 'User enabled.' : 'User disabled.');
    }

    /* -----------------------------------------------------------
     | POST /users/{user}/reset-password -> users.reset.password
     | Generates a temporary password and sets it
     ----------------------------------------------------------- */
    public function resetPassword(User $user)
    {
        if ($user->id === Auth::id()) {
            // চাইলে নিজ পাসওয়ার্ড রিসেট allow করতে পারেন; আমি রিসট্রিক্ট করলাম
            return back()->withErrors(['reset' => 'Please change your own password from Change Password page.']);
        }

        $temp = Str::random(10); // টেম্প পাসওয়ার্ড
        $user->password = $temp; // mutator hash করবে
        $user->password_reset_token = Str::random(32);
        $user->save();

        // প্রোডাকশনে ইমেইল/এসএমএস পাঠানো উত্তম; এখানে ব্যাক-এ দেখাচ্ছি
        return back()->with('status', "Temporary password set: {$temp}");
    }
}

