<?php

namespace App\Http\Controllers;

use App\Models\Unit;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class UnitController extends Controller
{
    /**
     * List page
     * GET /products/units  -> products.units.index
     */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));
        $units = Unit::query()
            ->when($q, fn ($qr) => $qr->where('name', 'like', "%{$q}%"))
            ->orderByDesc('id')
            ->paginate(20)
            ->withQueryString();

        return view('products.units.index', compact('units', 'q'));
    }

    /**
     * Create page
     * GET /products/units/create -> products.units.create
     */
    public function create()
    {
        return view('products.units.create');
    }

    /**
     * Store
     * POST /products/units -> products.units.store
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'              => ['required', 'string', 'max:191', 'unique:rm_units,name'],
            'conversation_size' => ['nullable', 'numeric', 'min:0'],
            // status string হিসেবে '0'/'1'
            'status'            => ['required', Rule::in(['0','1'])],
        ]);

        Unit::create([
            'name'              => $data['name'],
            'conversation_size' => $data['conversation_size'] ?? null,
            'status'            => $data['status'],   // '0' or '1'
            // created_by / updated_by Model booted() এ সেট হবে
        ]);

        return redirect()
            ->route('products.units.index')
            ->with('success', 'Unit created successfully.');
    }

    /**
     * Edit page
     * GET /products/units/{unit}/edit -> products.units.edit
     */
    public function edit(Unit $unit)
    {
        return view('products.units.edit', compact('unit'));
    }

    /**
     * Update
     * PUT/PATCH /products/units/{unit} -> products.units.update
     */
    public function update(Request $request, Unit $unit)
    {
        $data = $request->validate([
            'name'              => [
                'required', 'string', 'max:191',
                Rule::unique('rm_units', 'name')->ignore($unit->id),
            ],
            'conversation_size' => ['nullable', 'numeric', 'min:0'],
            'status'            => ['required', Rule::in(['0','1'])],
        ]);

        $unit->update([
            'name'              => $data['name'],
            'conversation_size' => $data['conversation_size'] ?? null,
            'status'            => $data['status'],
            // updated_by Model booted() এ সেট হবে
        ]);

        return redirect()
            ->route('products.units.index')
            ->with('success', 'Unit updated successfully.');
    }

    /**
     * Delete
     * DELETE /products/units/{unit} -> products.units.destroy
     */
    public function destroy(Unit $unit)
    {
        $unit->delete();

        return redirect()
            ->route('products.units.index')
            ->with('success', 'Unit deleted successfully.');
    }
}
