<?php

namespace App\Http\Controllers;

use App\Models\SalesDelivery;   // rm_deliveries
use App\Models\Company;
use App\Models\Sale;            // rm_sales
use App\Models\Customer;        // <-- added
use Illuminate\Http\Request;

class SalesDeliveryController extends Controller
{
    /**
     * List deliveries + filter dropdown data.
     */
    public function index()
    {
        $deliveries = SalesDelivery::with(['company', 'sale'])
            ->latest()
            ->paginate(20);

        // dropdown data for filters
        $companies  = Company::orderBy('name')->get();      // <-- added
        $customers  = Customer::orderBy('name')->get();     // <-- added

        // resources/views/sales_delivery/index.blade.php
        return view('sales_delivery.index', compact('deliveries', 'companies', 'customers'));
    }

    /**
     * Show create form.
     */
    public function create()
    {
        $companies = Company::orderBy('name')->get();
        $sales     = Sale::latest()->take(50)->get();

        // যদি create পাতায়ও customer dropdown লাগে, পাঠিয়ে দিন:
        $customers = Customer::orderBy('name')->get();      // <-- optional but safe

        // resources/views/sales_delivery/create.blade.php
        return view('sales_delivery.create', compact('companies', 'sales', 'customers'));
    }

    /**
     * Store into rm_deliveries.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'company_id'   => ['required', 'exists:companies,id'],
            'sale_id'      => ['required', 'exists:rm_sales,id'],
            'delivered_at' => ['nullable', 'date'],
            'note'         => ['nullable', 'string', 'max:500'],
        ]);

        $data['created_by'] = auth()->id();

        SalesDelivery::create($data);

        return redirect()
            ->route('sales_delivery.index')
            ->with('success', 'Delivery saved successfully.');
    }

    /**
     * Show one delivery.
     */
    public function show(SalesDelivery $salesDelivery)
    {
        return view('sales_delivery.show', ['delivery' => $salesDelivery]);
    }

    /**
     * Edit form.
     */
    public function edit(SalesDelivery $salesDelivery)
    {
        $companies = Company::orderBy('name')->get();
        $sales     = Sale::latest()->take(50)->get();
        $customers = Customer::orderBy('name')->get();      // <-- optional

        return view('sales_delivery.edit', compact('salesDelivery', 'companies', 'sales', 'customers'));
    }

    /**
     * Update.
     */
    public function update(Request $request, SalesDelivery $salesDelivery)
    {
        $data = $request->validate([
            'company_id'   => ['required', 'exists:companies,id'],
            'sale_id'      => ['required', 'exists:rm_sales,id'],
            'delivered_at' => ['nullable', 'date'],
            'note'         => ['nullable', 'string', 'max:500'],
        ]);

        $salesDelivery->update($data);

        return redirect()
            ->route('sales_delivery.index')
            ->with('success', 'Delivery updated.');
    }

    /**
     * Delete.
     */
    public function destroy(SalesDelivery $salesDelivery)
    {
        $salesDelivery->delete();

        return back()->with('success', 'Delivery deleted.');
    }

    /**
     * Optional: AJAX helper to populate "Sale Id" by company.
     */
    public function salesByCompany(Company $company)
    {
        $sales = Sale::where('company_id', $company->id)
            ->latest()
            ->select('id', 'invoice_no', 'created_at')
            ->get();

        return response()->json($sales);
    }
}
