<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Validation\Rule;
use Spatie\Permission\Models\Role;
use Spatie\Permission\Models\Permission;

class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    /** GET /access/roles -> roles.index */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $roles = Role::query()
            ->when($q !== '', function ($qr) use ($q) {
                $qr->where('name', 'like', "%{$q}%");
            })
            ->withCount('permissions')
            ->orderBy('name')
            ->paginate(20);

        return view('roles.index', compact('roles', 'q'));
    }

    /** GET /access/roles/create ->roles.create */
    public function create()
    {
        $permissions = Permission::orderBy('name')->get();

        return view('roles.create', compact('permissions'));
    }

    /** POST /access/roles -> roles.store */
    public function store(Request $request)
    {
        $validated = $request->validate([
            'name'         => ['required', 'string', 'max:255',
                Rule::unique('roles', 'name')->where(fn($q)=>$q->where('guard_name','web'))
            ],
            'permissions'  => ['array'],
            'permissions.*'=> ['integer'],
        ]);

        $role = Role::create([
            'name'       => $validated['name'],
            'guard_name' => 'web',
        ]);

        // Attach permissions (Spatie)
        if (!empty($validated['permissions'])) {
            $role->syncPermissions($validated['permissions']);
        }

        return redirect()->route('roles.index')->with('status', 'Role created.');
    }

    /** GET /access/roles/{id}/edit -> roles.edit */
    public function edit(string $id)
    {
        $role         = Role::findOrFail($id);
        $permissions  = Permission::orderBy('name')->get();
        $rolePermissions = $role->permissions()->pluck('id')->toArray();

        return view('roles.edit', compact('role', 'permissions', 'rolePermissions'));
    }

    /** PUT /access/roles/{id} -> access.roles.update */
    public function update(Request $request, string $id)
    {
        $role = Role::findOrFail($id);

        $validated = $request->validate([
            'name'         => ['required','string','max:255',
                Rule::unique('roles','name')
                    ->where(fn($q)=>$q->where('guard_name','web'))
                    ->ignore($role->id)
            ],
            'permissions'  => ['array'],
            'permissions.*'=> ['integer'],
        ]);

        $role->name = $validated['name'];
        // guard_name সাধারণত পরিবর্তন করা হয় না; তবু ensure করি:
        if (empty($role->guard_name)) {
            $role->guard_name = 'web';
        }
        $role->save();

        // Resync permissions
        $role->syncPermissions($validated['permissions'] ?? []);

        return redirect()->route('roles.index')->with('status', 'Role updated.');
    }

    /** DELETE /access/roles/{id} -> roles.destroy */
    public function destroy(string $id)
    {
        $role = Role::findOrFail($id);

        if (in_array(strtolower($role->name), ['super admin','admin'])) {
            return back()->withErrors(['delete' => 'Protected role cannot be deleted.']);
        }

        $role->delete();

        return redirect()->route('roles.index')->with('status', 'Role deleted.');
    }
}

