<?php

namespace App\Http\Controllers;

use App\Models\RawProductAdjustment;
use App\Models\Warehouse;
use App\Models\Product;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RawProductAdjustmentController extends Controller
{
    /**
     * List page — passes $adjustments to the blade.
     */
    public function index()
    {
        // Load with warehouse; paginate to match table footer if needed
        $adjustments = RawProductAdjustment::with('warehouse')
            ->latest('id')
            ->paginate(15); // or ->get()

        return view('adjustment.raw_product.index', compact('adjustments'));
    }

    /**
     * Create form — loads warehouses/products.
     */
    public function create()
    {
        $warehouses = Warehouse::where('status', 1)->get();
        $products   = Product::where('status', 1)->get();

        return view('adjustment.raw_product.create', compact('warehouses', 'products'));
    }

    /**
     * Store — saves into rm_product_adjustments (master).
     * Items table is optional; wire if/when you add it.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'warehouse_id'             => ['required', 'exists:warehouses,id'],
            'date'                     => ['required', 'date'],
            'adjust_type'              => ['required', 'in:IN,OUT,in,out'],
            'note'                     => ['nullable', 'string', 'max:1000'],
            'items'                    => ['required', 'array', 'min:1'],
            'items.*.product_id'       => ['required', 'exists:products,id'],
            'items.*.qty'              => ['required', 'numeric', 'min:0.01'],
            // 'items.*.price'          => ['nullable','numeric','min:0'], // if you use price
        ]);

        // Calculate total qty from items:
        $totalQty = 0;
        foreach ($data['items'] as $it) {
            $totalQty += (float) $it['qty'];
        }

        $adjustType = strtoupper($data['adjust_type']);

        DB::transaction(function () use ($request, $totalQty, $adjustType) {
            // ✅ master row into rm_product_adjustments
            $master = RawProductAdjustment::create([
                'warehouse_id' => $request->warehouse_id,
                'date'         => $request->date,
                'adjust_type'  => $adjustType, // IN / OUT
                'total_qty'    => $totalQty,
                'note'         => $request->note,
            ]);

            // If you have a detail table, save rows here (example):
            // foreach ($request->items as $it) {
            //     RawProductAdjustmentItem::create([
            //         'raw_product_adjustment_id' => $master->id,
            //         'product_id' => $it['product_id'],
            //         'qty'       => $it['qty'],
            //         'price'     => $it['price'] ?? 0,
            //         'subtotal'  => ($it['price'] ?? 0) * $it['qty'],
            //     ]);
            // }
        });

        return redirect()
            ->route('adjustment.raw_product.index')
            ->with('success', 'Raw product adjustment created.');
    }

    /**
     * Show (for view/print).
     */
    public function show($id)
    {
        $adjustment = RawProductAdjustment::with('warehouse')
            ->findOrFail($id);

        return view('adjustment.raw_product.show', compact('adjustment'));
    }

    /**
     * Edit form.
     */
    public function edit($id)
    {
        $adjustment = RawProductAdjustment::findOrFail($id);
        $warehouses = Warehouse::where('status', 1)->get();
        $products   = Product::where('status', 1)->get();

        return view('adjustment.raw_product.edit', compact('adjustment','warehouses','products'));
    }

    /**
     * Update master; extend to update items if you add a detail table.
     */
    public function update(Request $request, $id)
    {
        $adjustment = RawProductAdjustment::findOrFail($id);

        $data = $request->validate([
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'date'         => ['required', 'date'],
            'adjust_type'  => ['required', 'in:IN,OUT,in,out'],
            'note'         => ['nullable', 'string', 'max:1000'],
        ]);

        $adjustment->update([
            'warehouse_id' => $data['warehouse_id'],
            'date'         => $data['date'],
            'adjust_type'  => strtoupper($data['adjust_type']),
            'note'         => $data['note'] ?? null,
            // total_qty: recalc if you also edit items
        ]);

        return redirect()
            ->route('adjustment.raw_product.index')
            ->with('success', 'Raw product adjustment updated.');
    }

    /**
     * Delete master; also delete detail rows if you add them.
     */
    public function destroy($id)
    {
        $adjustment = RawProductAdjustment::findOrFail($id);

        DB::transaction(function () use ($adjustment) {
            // RawProductAdjustmentItem::where('raw_product_adjustment_id', $adjustment->id)->delete();
            $adjustment->delete();
        });

        return redirect()
            ->route('adjustment.raw_product.index')
            ->with('success', 'Raw product adjustment deleted.');
    }
}
