<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;

use App\Models\Company;
use App\Models\Supplier;
use App\Models\Employee;
use App\Models\Warehouse;
use App\Models\RmProduct as Product; // ✅ alias RmProduct -> Product
use App\Models\Bank;

class RawMaterialPurchaseController extends Controller
{
    /**
     * GET /purchase/raw-material
     */
    public function index()
    {
        return view('purchase.raw_material.index', [
            // 'rows' => RawMaterialPurchase::latest()->paginate(20),
        ]);
    }

    /**
     * GET /purchase/raw-material/create
     * Show the create form (send dropdown data).
     */
    public function create()
    {
        $companies  = Company::orderBy('name')->get(['id','name']);
        $suppliers  = Supplier::orderBy('name')->get(['id','name']);
        $employees  = Employee::orderBy('name')->get(['id','name']);
        $warehouses = Warehouse::orderBy('name')->get(['id','name']);

        // ✅ pulls from rm_products table via RmProduct model
        $products   = Product::orderBy('name')->get(['id','name','kg_per_bag']);

        $banks      = Bank::orderBy('name')->get(['id','name']);

        return view('purchase.raw_material.create', compact(
            'companies','suppliers','employees','warehouses','products','banks'
        ));
    }

    /**
     * POST /purchase/raw-material
     */
    public function store(Request $request)
    {
        $request->validate([
            'company_id'   => ['required'],
            'supplier_id'  => ['required'],
            'date'         => ['required', 'date'],
            'items'        => ['required', 'array', 'min:1'],
            'items.*.product_id' => ['required'],
            'items.*.price'      => ['nullable', 'numeric', 'min:0'],
            'items.*.kg'         => ['nullable', 'numeric', 'min:0'],
            'items.*.bag'        => ['nullable', 'integer', 'min:0'],
            'items.*.mon'        => ['nullable', 'numeric', 'min:0'],
            'discount'     => ['nullable', 'numeric', 'min:0'],
            'paid'         => ['nullable', 'numeric', 'min:0'],
        ]);

        return redirect()
            ->route('purchase.raw_material.index')
            ->with('success', 'Raw material purchase created.');
    }

    /**
     * GET /purchase/raw-material/{raw_material}
     */
    public function show(string $raw_material)
    {
        return view('purchase.raw_material.show', [
            'purchase' => $raw_material,
        ]);
    }

    /**
     * GET /purchase/raw-material/{raw_material}/edit
     */
    public function edit(string $raw_material)
    {
        $companies  = Company::orderBy('name')->get(['id','name']);
        $suppliers  = Supplier::orderBy('name')->get(['id','name']);
        $employees  = Employee::orderBy('name')->get(['id','name']);
        $warehouses = Warehouse::orderBy('name')->get(['id','name']);

        // ✅ same alias here
        $products   = Product::orderBy('name')->get(['id','name','kg_per_bag']);

        $banks      = Bank::orderBy('name')->get(['id','name']);

        return view('purchase.raw_material.edit', [
            'purchase'   => $raw_material,
            'companies'  => $companies,
            'suppliers'  => $suppliers,
            'employees'  => $employees,
            'warehouses' => $warehouses,
            'products'   => $products,
            'banks'      => $banks,
        ]);
    }

    /**
     * PUT /purchase/raw-material/{raw_material}
     */
    public function update(Request $request, string $raw_material)
    {
        $request->validate([
            'company_id'   => ['sometimes', 'required'],
            'supplier_id'  => ['sometimes', 'required'],
            'date'         => ['sometimes', 'required', 'date'],
            'items'        => ['sometimes', 'array', 'min:1'],
            'items.*.product_id' => ['required_with:items'],
            'items.*.price'      => ['nullable', 'numeric', 'min:0'],
            'items.*.kg'         => ['nullable', 'numeric', 'min:0'],
            'items.*.bag'        => ['nullable', 'integer', 'min:0'],
            'items.*.mon'        => ['nullable', 'numeric', 'min:0'],
            'discount'     => ['nullable', 'numeric', 'min:0'],
            'paid'         => ['nullable', 'numeric', 'min:0'],
        ]);

        return redirect()
            ->route('purchase.raw_material.index')
            ->with('success', 'Raw material purchase updated.');
    }

    /**
     * DELETE /purchase/raw-material/{raw_material}
     */
    public function destroy(string $raw_material)
    {
        return redirect()
            ->route('purchase.raw_material.index')
            ->with('success', 'Raw material purchase deleted.');
    }

    /**
     * GET /purchase/raw-material/{raw_material}/print
     */
    public function print(string $raw_material)
    {
        return view('purchase.raw_material.print', [
            'purchase' => $raw_material,
        ]);
    }
}
