<?php

namespace App\Http\Controllers;

use App\Models\RawMaterial;
use App\Models\Category;   // rm_categories
use App\Models\Supplier;   // rm_suppliers
use App\Models\Warehouse;  // warehouses
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class RawMaterialController extends Controller
{
    /** GET /products/raw-products */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $products = RawMaterial::with(['category', 'warehouse', 'supplier'])
            ->when($q, function ($qr) use ($q) {
                $qr->where(function ($w) use ($q) {
                    $w->where('name', 'like', "%{$q}%")
                      ->orWhereHas('category', fn ($c) => $c->where('name', 'like', "%{$q}%"));
                });
            })
            ->where('type', 1) // 1 => raw (tinyint)
            ->orderBy('id', 'asc')
            ->paginate(12)
            ->withQueryString();

        return view('products.raw-materials.index', compact('products', 'q'));
    }

    /** GET /products/raw-products/create */
    public function create()
    {
        $categories = Category::orderBy('name')->get(['id', 'name']);
        $suppliers  = Supplier::orderBy('name')->get(['id', 'name']);
        $warehouses = Warehouse::orderBy('name')->get(['id', 'name']);

        return view('products.raw-materials.create', compact('categories', 'suppliers', 'warehouses'));
    }

    /** POST /products/raw-products */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'         => ['required', 'string', 'max:191'],
            'category_id'  => ['required', 'exists:rm_categories,id'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'supplier_id'  => ['nullable', 'exists:rm_suppliers,id'],
            'opening_qty'  => ['nullable', 'numeric'], // request field → opening_quantity
            'alert_qty'    => ['nullable', 'numeric'], // request field → alert_quantity
            'vat'          => ['nullable', 'numeric'],
            'price'        => ['nullable', 'numeric'], // request field → product_price
            'status'       => ['required', Rule::in(['0', '1'])],
            'description'  => ['nullable', 'string'],
            'image'        => ['nullable', 'image', 'max:2048'],
        ]);

        // Upload image
        $imagePath = null;
        if ($request->hasFile('image')) {
            $stored    = $request->file('image')->store('products', 'public'); // storage/app/public/products
            $imagePath = 'storage/' . $stored;                                  // public path to save in DB
        }

        RawMaterial::create([
            'name'               => $data['name'],
            'category_id'        => $data['category_id'],
            'warehouse_id'       => $data['warehouse_id'],
            'supplier_id'        => $data['supplier_id'] ?? null,
            'opening_quantity'   => $data['opening_qty'] ?? 0,
            'alert_quantity'     => $data['alert_qty']   ?? 0,
            'vat'                => $data['vat']         ?? null,
            'product_price'      => $data['price']       ?? null,
            'status'             => (int) $data['status'],
            'type'               => 1, // raw
            'image'              => $imagePath,
            'description'        => $data['description'] ?? null,
            'created_by'         => auth()->id(),
            'updated_by'         => auth()->id(),
        ]);

        return redirect()
            ->route('products.raw-materials.index')
            ->with('success', 'Product created successfully.');
    }

    /** GET /products/raw-products/{product}/edit */
    public function edit(RawMaterial $product)
    {
        abort_unless((int)$product->type === 1, 404);

        $categories = Category::orderBy('name')->get(['id', 'name']);
        $suppliers  = Supplier::orderBy('name')->get(['id', 'name']);
        $warehouses = Warehouse::orderBy('name')->get(['id', 'name']);

        return view('products.raw-materials.edit', compact('product', 'categories', 'suppliers', 'warehouses'));
    }

    /** PUT /products/raw-products/{product} */
    public function update(Request $request, RawMaterial $product)
    {
        abort_unless((int)$product->type === 1, 404);

        $data = $request->validate([
            'name'         => ['required', 'string', 'max:191'],
            'category_id'  => ['required', 'exists:rm_categories,id'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'supplier_id'  => ['nullable', 'exists:rm_suppliers,id'],
            'opening_qty'  => ['nullable', 'numeric'],
            'alert_qty'    => ['nullable', 'numeric'],
            'vat'          => ['nullable', 'numeric'],
            'price'        => ['nullable', 'numeric'],
            'status'       => ['required', Rule::in(['0', '1'])],
            'description'  => ['nullable', 'string'],
            'image'        => ['nullable', 'image', 'max:2048'],
        ]);

        // Replace image if new uploaded
        $imagePath = $product->image;
        if ($request->hasFile('image')) {
            if ($imagePath && str_starts_with($imagePath, 'storage/')) {
                $old = str_replace('storage/', '', $imagePath);
                Storage::disk('public')->delete($old);
            }
            $stored    = $request->file('image')->store('products', 'public');
            $imagePath = 'storage/' . $stored;
        }

        $product->update([
            'name'               => $data['name'],
            'category_id'        => $data['category_id'],
            'warehouse_id'       => $data['warehouse_id'],
            'supplier_id'        => $data['supplier_id'] ?? null,
            'opening_quantity'   => $data['opening_qty'] ?? $product->opening_quantity,
            'alert_quantity'     => $data['alert_qty']   ?? $product->alert_quantity,
            'vat'                => $data['vat']         ?? $product->vat,
            'product_price'      => $data['price']       ?? $product->product_price,
            'status'             => (int) $data['status'],
            'image'              => $imagePath,
            'description'        => $data['description'] ?? $product->description,
            'updated_by'         => auth()->id(),
        ]);

        return redirect()
            ->route('products.raw-materials.index')
            ->with('success', 'Product updated successfully.');
    }

    /** DELETE /products/raw-products/{product} */
    public function destroy(RawMaterial $product)
    {
        abort_unless((int)$product->type === 1, 404);

        if ($product->image && str_starts_with($product->image, 'storage/')) {
            $old = str_replace('storage/', '', $product->image);
            Storage::disk('public')->delete($old);
        }

        $product->delete();

        return redirect()
            ->route('products.raw-materials.index')
            ->with('success', 'Product deleted successfully.');
    }
}


