<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Warehouse;
use App\Models\RawBagOpening;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RawBagOpeningController extends Controller
{
    /**
     * Display a listing of the resource.
     * Supports simple filters: company_id, warehouse_id.
     */
    public function index(Request $request)
    {
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        $openings = RawBagOpening::with(['company', 'warehouse'])
            ->when($request->filled('company_id'), fn ($q) =>
                $q->where('company_id', $request->company_id))
            ->when($request->filled('warehouse_id'), fn ($q) =>
                $q->where('warehouse_id', $request->warehouse_id))
            ->latest('id')
            ->paginate(15);

        return view('adjustment.raw_bag_opening.index', compact('companies', 'warehouses', 'openings'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        return view('adjustment.raw_bag_opening.create', compact('companies', 'warehouses'));
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'company_id'   => ['required', 'exists:companies,id'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'opening_qty'  => ['required', 'numeric', 'min:0.01'],
            'note'         => ['nullable', 'string', 'max:1000'],
            // If you add a visible date field on the form, change the rule below to 'required|date'
            'date'         => ['nullable', 'date'],
        ]);

        // If date isn't provided, default to today.
        $data['date'] = $data['date'] ?? now()->toDateString();

        DB::transaction(function () use ($data) {
            RawBagOpening::create($data);
            // If you maintain stock ledgers, insert corresponding
            // stock rows here (omitted as out of scope for this request).
        });

        return redirect()
            ->route('adjustment.raw_bag_opening.index')
            ->with('success', 'Raw bag opening created successfully.');
    }

    /**
     * Display the specified resource.
     */
    public function show($id)
    {
        $opening = RawBagOpening::with(['company', 'warehouse'])->findOrFail($id);

        return view('adjustment.raw_bag_opening.show', compact('opening'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit($id)
    {
        $opening    = RawBagOpening::findOrFail($id);
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        return view('adjustment.raw_bag_opening.edit', compact('opening', 'companies', 'warehouses'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, $id)
    {
        $opening = RawBagOpening::findOrFail($id);

        $data = $request->validate([
            'company_id'   => ['required', 'exists:companies,id'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'opening_qty'  => ['required', 'numeric', 'min:0.01'],
            'note'         => ['nullable', 'string', 'max:1000'],
            'date'         => ['nullable', 'date'],
        ]);

        $data['date'] = $data['date'] ?? $opening->date ?? now()->toDateString();

        DB::transaction(function () use ($opening, $data) {
            $opening->update($data);
            // If you keep stock ledgers, adjust those here as well.
        });

        return redirect()
            ->route('adjustment.raw_bag_opening.index')
            ->with('success', 'Raw bag opening updated successfully.');
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy($id)
    {
        $opening = RawBagOpening::findOrFail($id);

        DB::transaction(function () use ($opening) {
            // Delete any dependent rows (e.g., stock ledger) here if you have them.
            $opening->delete();
        });

        return redirect()
            ->route('adjustment.raw_bag_opening.index')
            ->with('success', 'Raw bag opening deleted successfully.');
    }
}

