<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\RawBagAdjustment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class RawBagAdjustmentController extends Controller
{
    /** List */
    public function index(Request $request)
    {
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        $q = RawBagAdjustment::with(['company', 'warehouse'])->latest('id');

        if ($request->filled('company_id'))   $q->where('company_id',   $request->company_id);
        if ($request->filled('warehouse_id')) $q->where('warehouse_id', $request->warehouse_id);
        if ($request->filled('from'))         $q->whereDate('date', '>=', $request->from);
        if ($request->filled('to'))           $q->whereDate('date', '<=', $request->to);

        $adjustments = $q->paginate(15);

        return view('adjustment.raw_bag.index', compact('companies', 'warehouses', 'adjustments'));
    }

    /** Create form */
    public function create()
    {
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        // FIX: stop using a non-existent `type` column.
        // If you DO have a way to identify raw bags (e.g. category_id), filter there.
        // Example: ->where('category_id', BAG_CATEGORY_ID)
        $products = Product::where('status', 1)
            ->orderBy('name')
            ->get();

        return view('adjustment.raw_bag.create', compact('companies', 'warehouses', 'products'));
    }

    /** Store master (aggregate) */
    public function store(Request $request)
    {
        $data = $request->validate([
            'company_id'           => ['required', 'exists:companies,id'],
            'warehouse_id'         => ['required', 'exists:warehouses,id'],
            'date'                 => ['required', 'date'],
            'note'                 => ['nullable', 'string', 'max:1000'],

            'items'                => ['required', 'array', 'min:1'],
            'items.*.product_id'   => ['required', 'exists:products,id'],
            'items.*.qty'          => ['required', 'numeric', 'min:0.01'],
            'items.*.adjust_type'  => ['required', 'in:IN,OUT,in,out'],
        ]);

        $totalQty = 0.0;
        foreach ($data['items'] as $row) {
            $totalQty += (float) $row['qty'];
        }

        DB::transaction(function () use ($request, $totalQty) {
            RawBagAdjustment::create([
                'company_id'   => $request->company_id,
                'warehouse_id' => $request->warehouse_id,
                'date'         => $request->date,
                'total_qty'    => $totalQty,
                'note'         => $request->note,
            ]);

            // If you later add a detail/items table, insert rows here.
        });

        return redirect()->route('adjustment.raw_bag.index')
            ->with('success', 'Raw bag adjustment created.');
    }

    /** Show (optional) */
    public function show($id)
    {
        $adjustment = RawBagAdjustment::with(['company', 'warehouse'])->findOrFail($id);
        return view('adjustment.raw_bag.show', compact('adjustment'));
    }

    /** Edit form */
    public function edit($id)
    {
        $adjustment = RawBagAdjustment::findOrFail($id);
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        // Same fix as in create()
        $products = Product::where('status', 1)->orderBy('name')->get();

        return view('adjustment.raw_bag.edit', compact('adjustment', 'companies', 'warehouses', 'products'));
    }

    /** Update master */
    public function update(Request $request, $id)
    {
        $adjustment = RawBagAdjustment::findOrFail($id);

        $data = $request->validate([
            'company_id'   => ['required', 'exists:companies,id'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'date'         => ['required', 'date'],
            'note'         => ['nullable', 'string', 'max:1000'],
        ]);

        $adjustment->update($data);

        return redirect()->route('adjustment.raw_bag.index')
            ->with('success', 'Raw bag adjustment updated.');
    }

    /** Destroy */
    public function destroy($id)
    {
        $adjustment = RawBagAdjustment::findOrFail($id);

        DB::transaction(function () use ($adjustment) {
            // If you add an items table, delete items first.
            $adjustment->delete();
        });

        return redirect()->route('adjustment.raw_bag.index')
            ->with('success', 'Raw bag adjustment deleted.');
    }
}

