<?php

namespace App\Http\Controllers;

use App\Models\OrderDelivery;  // -> rm_order_deliveries
use App\Models\Order;          // -> rm_orders
use App\Models\Company;
use App\Models\Customer;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\View\View;

class OrderDeliveryController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth']);
    }

    /**
     * List + filters
     */
    public function index(Request $request): View
    {
        $q = OrderDelivery::query()
            ->when($request->filled('company_id'), fn($qr) =>
                $qr->where('company_id', (int) $request->company_id)
            )
            ->when($request->filled('customer_id'), fn($qr) =>
                $qr->where('customer_id', (int) $request->customer_id)
            )
            ->when($request->filled('order_id'), fn($qr) =>
                $qr->where('order_id', (int) $request->order_id)
            )
            ->when($request->filled('date_from'), function ($qr) use ($request) {
                if (Schema::hasColumn('rm_order_deliveries', 'delivery_date')) {
                    $qr->whereDate('delivery_date', '>=', $request->date_from);
                } else {
                    $qr->whereDate('created_at', '>=', $request->date_from);
                }
            })
            ->when($request->filled('date_to'), function ($qr) use ($request) {
                if (Schema::hasColumn('rm_order_deliveries', 'delivery_date')) {
                    $qr->whereDate('delivery_date', '<=', $request->date_to);
                } else {
                    $qr->whereDate('created_at', '<=', $request->date_to);
                }
            })
            ->latest('id');

        $deliveries = $q->paginate(20)->withQueryString();

        $companies = Company::select('id','name')->orderBy('name')->get();
        $customers = Customer::select('id','name')->orderBy('name')->get();
        $orders    = Order::select('id')->latest('id')->limit(200)->get(); // dropdown-এর জন্য

        return view('order_delivery.index', compact('deliveries','companies','customers','orders'));
    }

    /**
     * Create form
     */
    public function create(): View
    {
        $companies = Company::select('id','name')->orderBy('name')->get();
        $customers = Customer::select('id','name')->orderBy('name')->get();
        $orders    = Order::select('id')->latest('id')->limit(200)->get();

        return view('order_delivery.create', compact('companies','customers','orders'));
    }

    /**
     * Store
     */
    public function store(Request $request): RedirectResponse
    {
        // delivery_date নাম নাও বা form থেকে date এলেও ধরি
        $request->merge([
            '__delivery_date' => $request->input('delivery_date') ?? $request->input('date'),
        ]);

        // Basic validation (ন্যূনতম)
        $request->validate([
            'company_id'  => ['required','integer','min:1'],
            'customer_id' => ['required','integer','min:1'],
            'order_id'    => ['required','integer','min:1'],
        ]);

        // Insertable payload (ডাইনামিক)
        $data = [
            'company_id'  => (int) $request->company_id,
            'customer_id' => (int) $request->customer_id,
            'order_id'    => (int) $request->order_id,
        ];

        // Optional columns থাকলে যোগ করো
        if (Schema::hasColumn('rm_order_deliveries', 'delivery_date') && $request->__delivery_date) {
            $data['delivery_date'] = $request->__delivery_date;
        }
        if (Schema::hasColumn('rm_order_deliveries', 'note') && $request->filled('note')) {
            $data['note'] = (string) $request->note;
        }
        if (Schema::hasColumn('rm_order_deliveries', 'delivered_by')) {
            $data['delivered_by'] = auth()->id();
        }
        if (Schema::hasColumn('rm_order_deliveries', 'total_qty') && $request->filled('total_qty')) {
            $data['total_qty'] = (float) $request->total_qty;
        }
        if (Schema::hasColumn('rm_order_deliveries', 'total_amount') && $request->filled('total_amount')) {
            $data['total_amount'] = (float) $request->total_amount;
        }
        if (Schema::hasColumn('rm_order_deliveries', 'status') && $request->filled('status')) {
            $data['status'] = (string) $request->status; // e.g. delivered/pending
        }

        OrderDelivery::create($data);

        return redirect()
            ->route('order_delivery.index')
            ->with('success', 'Delivery order created.');
    }

    /**
     * Show single delivery (optional)
     */
    public function show(int $id): View
    {
        $delivery = OrderDelivery::findOrFail($id);
        return view('order_delivery.show', compact('delivery'));
    }

    /**
     * Edit form
     */
    public function edit(int $id): View
    {
        $delivery  = OrderDelivery::findOrFail($id);
        $companies = Company::select('id','name')->orderBy('name')->get();
        $customers = Customer::select('id','name')->orderBy('name')->get();
        $orders    = Order::select('id')->latest('id')->limit(200)->get();

        return view('order_delivery.edit', compact('delivery','companies','customers','orders'));
    }

    /**
     * Update
     */
    public function update(Request $request, int $id): RedirectResponse
    {
        $delivery = OrderDelivery::findOrFail($id);

        $request->merge([
            '__delivery_date' => $request->input('delivery_date') ?? $request->input('date'),
        ]);

        $request->validate([
            'company_id'  => ['required','integer','min:1'],
            'customer_id' => ['required','integer','min:1'],
            'order_id'    => ['required','integer','min:1'],
        ]);

        $data = [
            'company_id'  => (int) $request->company_id,
            'customer_id' => (int) $request->customer_id,
            'order_id'    => (int) $request->order_id,
        ];

        if (Schema::hasColumn('rm_order_deliveries', 'delivery_date') && $request->__delivery_date) {
            $data['delivery_date'] = $request->__delivery_date;
        }
        if (Schema::hasColumn('rm_order_deliveries', 'note')) {
            $data['note'] = (string) ($request->note ?? '');
        }
        if (Schema::hasColumn('rm_order_deliveries', 'total_qty')) {
            $data['total_qty'] = (float) ($request->total_qty ?? 0);
        }
        if (Schema::hasColumn('rm_order_deliveries', 'total_amount')) {
            $data['total_amount'] = (float) ($request->total_amount ?? 0);
        }
        if (Schema::hasColumn('rm_order_deliveries', 'status') && $request->filled('status')) {
            $data['status'] = (string) $request->status;
        }

        $delivery->update($data);

        return redirect()
            ->route('order_delivery.index')
            ->with('success', 'Delivery order updated.');
    }

    /**
     * Delete
     */
    public function destroy(int $id): RedirectResponse
    {
        $delivery = OrderDelivery::findOrFail($id);
        $delivery->delete();

        return redirect()
            ->route('order_delivery.index')
            ->with('success', 'Delivery order deleted.');
    }
}

