<?php

namespace App\Http\Controllers;

use App\Models\Order;
use App\Models\OrderItem;
use App\Models\Company;
use App\Models\Customer;
use App\Models\Employee;
use App\Models\Product;
use App\Models\Category;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\View\View;

class OrderController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth']);
    }

    /** Order List + Filters */
    public function index(Request $request): View
    {
        $q = Order::query()
            ->when($request->filled('company_id'), fn($qr) =>
                $qr->where('company_id', (int)$request->company_id)
            )
            ->when($request->filled('customer_id'), fn($qr) =>
                $qr->where('customer_id', (int)$request->customer_id)
            )
            ->when($request->filled('date_from'), fn($qr) =>
                $qr->whereDate('order_date', '>=', $request->date_from)
            )
            ->when($request->filled('date_to'), fn($qr) =>
                $qr->whereDate('order_date', '<=', $request->date_to)
            )
            ->with(['company:id,name', 'customer:id,name', 'employee:id,name'])
            ->latest('order_date')->latest('id');

        $orders = $q->paginate(20)->withQueryString();

        $companies = Company::select('id','name')->orderBy('name')->get();
        $customers = Customer::select('id','name')->orderBy('name')->get();

        return view('order.index', compact('orders','companies','customers'));
    }

    /** Show Create Form */
    public function create(): View
    {
        $companies  = Company::select('id','name')->orderBy('name')->get();
        $customers  = Customer::select('id','name')->orderBy('name')->get();
        $employees  = Employee::select('id','name')->orderBy('name')->get();
        $categories = Category::select('id','name')->orderBy('name')->get();

        // Flexible product columns
        $nameCol = Schema::hasColumn('products','name')
            ? 'name' : (Schema::hasColumn('products','product_name') ? 'product_name' : null);

        $unitCol = null;
        foreach (['unit_name','unit','uom_name','uom'] as $c) {
            if (Schema::hasColumn('products', $c)) { $unitCol = $c; break; }
        }

        $selects = ['id','category_id'];
        if ($nameCol) $selects[] = $nameCol === 'name' ? 'name' : DB::raw("$nameCol as name");
        if ($unitCol) $selects[] = DB::raw("$unitCol as unit_name");
        if (Schema::hasColumn('products','price'))         $selects[] = 'price';
        if (Schema::hasColumn('products','available_qty')) $selects[] = 'available_qty';

        $products = Product::select($selects)->with(['category:id,name'])->orderBy($nameCol ?: 'id')->get();
        foreach ($products as $p) {
            $p->price         = $p->price ?? 0;
            $p->available_qty = $p->available_qty ?? 0;
        }

        return view('order.create', compact('companies','customers','employees','categories','products'));
    }

    /** Store Order + Items */
    public function store(Request $request): RedirectResponse
    {
        // accept either 'order_date' or 'date' from form
        $request->merge([
            '__order_date' => $request->input('order_date') ?? $request->input('date'),
        ]);

        $request->validate([
            'company_id'    => ['required','integer','min:1'],
            'customer_id'   => ['required','integer','min:1'],
            'employee_id'   => ['nullable','integer','min:1'],
            '__order_date'  => ['required','date'],
            'delivery_date' => ['nullable','date'],
            'customer_due'  => ['nullable','numeric','min:0'],
        ]);

        $lines = $request->input('items', []);
        if (!is_array($lines) || count($lines) === 0) {
            return back()->withErrors(['items' => 'Please add at least one product line.'])->withInput();
        }
        foreach ($lines as $i => $line) {
            $request->validate([
                "items.$i.product_id"  => ['required','integer','min:1'],
                "items.$i.category_id" => ['nullable','integer','min:1'],
                "items.$i.qty"         => ['required','numeric','min:0.0001'],
                "items.$i.price"       => ['required','numeric','min:0'],
            ]);
        }

        $totalQty = 0; $grandTotal = 0;
        foreach ($lines as $line) {
            $q   = (float)($line['qty'] ?? 0);
            $pr  = (float)($line['price'] ?? 0);
            $totalQty   += $q;
            $grandTotal += $q * $pr;
        }

        $userId = auth()->id();

        DB::transaction(function () use ($request, $lines, $totalQty, $grandTotal, $userId) {
            $order = Order::create([
                'company_id'    => (int)$request->company_id,
                'customer_id'   => (int)$request->customer_id,
                'employee_id'   => $request->filled('employee_id') ? (int)$request->employee_id : null,
                'order_date'    => $request->__order_date,   // ✅ rm_orders.order_date
                'delivery_date' => $request->delivery_date ?: null,
                'customer_due'  => (float)($request->customer_due ?? 0),
                'total_qty'     => round($totalQty, 3),
                'grand_total'   => round($grandTotal, 2),
                'created_by'    => $userId,
            ]);

            foreach ($lines as $line) {
                $qty   = (float)$line['qty'];
                $price = (float)$line['price'];

                OrderItem::create([
                    'order_id'    => $order->id,
                    'product_id'  => (int)$line['product_id'],
                    'category_id' => !empty($line['category_id']) ? (int)$line['category_id'] : null,
                    'qty'         => round($qty, 3),
                    'price'       => round($price, 2),
                    'subtotal'    => round($qty * $price, 2),
                ]);
            }
        });

        return redirect()->route('order.index')->with('success','Order created successfully.');
    }

    /** Show */
    public function show(Order $order): View
    {
        $order->load(['company:id,name','customer:id,name','employee:id,name','items.product:id,name']);
        return view('order.show', compact('order'));
    }

    /** Edit */
    public function edit(Order $order): View
    {
        $order->load(['items']);

        $companies  = Company::select('id','name')->orderBy('name')->get();
        $customers  = Customer::select('id','name')->orderBy('name')->get();
        $employees  = Employee::select('id','name')->orderBy('name')->get();
        $categories = Category::select('id','name')->orderBy('name')->get();

        $nameCol = Schema::hasColumn('products','name') ? 'name' : (Schema::hasColumn('products','product_name') ? 'product_name' : null);
        $unitCol = null;
        foreach (['unit_name','unit','uom_name','uom'] as $c) if (Schema::hasColumn('products',$c)) { $unitCol=$c; break; }

        $selects = ['id','category_id'];
        if ($nameCol) $selects[] = $nameCol === 'name' ? 'name' : DB::raw("$nameCol as name");
        if ($unitCol) $selects[] = DB::raw("$unitCol as unit_name");
        if (Schema::hasColumn('products','price'))         $selects[] = 'price';
        if (Schema::hasColumn('products','available_qty')) $selects[] = 'available_qty';

        $products = Product::select($selects)->with(['category:id,name'])->orderBy($nameCol ?: 'id')->get();
        foreach ($products as $p) { $p->price = $p->price ?? 0; $p->available_qty = $p->available_qty ?? 0; }

        return view('order.edit', compact('order','companies','customers','employees','categories','products'));
    }

    /** Update */
    public function update(Request $request, Order $order): RedirectResponse
    {
        $request->merge([
            '__order_date' => $request->input('order_date') ?? $request->input('date'),
        ]);

        $request->validate([
            'company_id'    => ['required','integer','min:1'],
            'customer_id'   => ['required','integer','min:1'],
            'employee_id'   => ['nullable','integer','min:1'],
            '__order_date'  => ['required','date'],
            'delivery_date' => ['nullable','date'],
            'customer_due'  => ['nullable','numeric','min:0'],
        ]);

        $lines = $request->input('items', []);
        if (!is_array($lines) || count($lines) === 0) {
            return back()->withErrors(['items' => 'Please add at least one product line.'])->withInput();
        }
        foreach ($lines as $i => $line) {
            $request->validate([
                "items.$i.product_id"  => ['required','integer','min:1'],
                "items.$i.category_id" => ['nullable','integer','min:1'],
                "items.$i.qty"         => ['required','numeric','min:0.0001'],
                "items.$i.price"       => ['required','numeric','min:0'],
            ]);
        }

        $totalQty = 0; $grandTotal = 0;
        foreach ($lines as $line) {
            $q = (float)$line['qty']; $pr = (float)$line['price'];
            $totalQty += $q; $grandTotal += $q * $pr;
        }

        DB::transaction(function () use ($request, $order, $lines, $totalQty, $grandTotal) {
            $order->update([
                'company_id'    => (int)$request->company_id,
                'customer_id'   => (int)$request->customer_id,
                'employee_id'   => $request->filled('employee_id') ? (int)$request->employee_id : null,
                'order_date'    => $request->__order_date,
                'delivery_date' => $request->delivery_date ?: null,
                'customer_due'  => (float)($request->customer_due ?? 0),
                'total_qty'     => round($totalQty, 3),
                'grand_total'   => round($grandTotal, 2),
            ]);

            OrderItem::where('order_id', $order->id)->delete();

            foreach ($lines as $line) {
                $q = (float)$line['qty']; $pr = (float)$line['price'];
                OrderItem::create([
                    'order_id'    => $order->id,
                    'product_id'  => (int)$line['product_id'],
                    'category_id' => !empty($line['category_id']) ? (int)$line['category_id'] : null,
                    'qty'         => round($q, 3),
                    'price'       => round($pr, 2),
                    'subtotal'    => round($q * $pr, 2),
                ]);
            }
        });

        return redirect()->route('order.index')->with('success','Order updated.');
    }

    /** Delete */
    public function destroy(Order $order): RedirectResponse
    {
        DB::transaction(function () use ($order) {
            OrderItem::where('order_id', $order->id)->delete();
            $order->delete();
        });

        return redirect()->route('order.index')->with('success','Order deleted.');
    }
}
