<?php

namespace App\Http\Controllers\Install;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Str;
use Nwidart\Modules\Facades\Module;
use ZipArchive;

class ModulesController extends Controller
{
    public function index()
    {
        $modules = Module::toCollection()->map(function ($m) {
            $name = $m['name'];
            $enabled = Module::find($name)->isEnabled();
            $baseNs = "\\Modules\\{$name}\\Http\\Controllers\\InstallController";
            return [
                'name'          => $name,
                'alias'         => $m['alias'] ?? $name,
                'version'       => $m['version'] ?? null,
                'enabled'       => $enabled,
                'install_link'  => class_exists($baseNs) ? action("{$baseNs}@index") : '#',
                'update_link'   => class_exists($baseNs) ? action("{$baseNs}@update") : '#',
                'uninstall_link'=> class_exists($baseNs) ? action("{$baseNs}@uninstall") : '#',
                'path'          => Module::find($name)->getPath(),
            ];
        })->values();

        return view('install.modules.index', compact('modules'));
    }

    public function regenerate()
    {
        Artisan::call('module:publish');
        Artisan::call('optimize:clear');
        Artisan::call('config:cache');
        Artisan::call('route:cache');
        return back()->with('status', ['success' => 1, 'msg' => 'Regenerated.']);
    }

    public function update(Request $request, string $module_name)
    {
        $module = Module::find($module_name);
        abort_unless($module, 404);

        if ($request->action_type === 'activate') {
            $module->enable();
        } elseif ($request->action_type === 'deactivate') {
            $module->disable();
        }

        return back()->with('status', ['success' => 1, 'msg' => 'Success']);
    }

    public function destroy(string $module_name)
    {
        $module = Module::find($module_name);
        abort_unless($module, 404);

        $path = $module->getPath();
        $module->disable();
        File::deleteDirectory($path);

        Artisan::call('optimize:clear');

        return back()->with('status', ['success' => 1, 'msg' => 'Module deleted']);
    }

    public function uploadModule(Request $request)
    {
        $request->validate(['module' => ['required', 'file', 'mimes:zip']]);

        $zipFile = $request->file('module');
        $moduleName = Str::of($zipFile->getClientOriginalName())->replace('.zip', '')->trim()->toString();

        $modulesDir = base_path('Modules');
        if (! is_dir($modulesDir)) {
            mkdir($modulesDir, 0777, true);
        }

        $zip = new ZipArchive();
        abort_unless($zip->open($zipFile) === true, 422, 'Invalid zip');

        $zip->extractTo($modulesDir . DIRECTORY_SEPARATOR);
        $zip->close();

        $extractedPath = $modulesDir . DIRECTORY_SEPARATOR . $moduleName;

        if (
            ! file_exists($extractedPath . '/composer.json') ||
            ! file_exists($extractedPath . '/module.json')
        ) {
            File::deleteDirectory($extractedPath);
            return back()->with('status', ['success' => 0, 'msg' => 'Invalid module structure']);
        }

        exec(PHP_BINARY . ' ' . base_path('artisan') . ' optimize:clear');
        exec('composer dump-autoload -d ' . base_path());

        return back()->with('status', ['success' => 1, 'msg' => 'Module uploaded']);
    }
}
