<?php

namespace App\Http\Controllers;

use App\Models\GSAdjustment;
use App\Models\Company;
use App\Models\Warehouse;
use App\Models\Product;

use Illuminate\Http\Request;
use Illuminate\Http\RedirectResponse;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;
use Illuminate\View\View;

class GSAdjustmentController extends Controller
{
    public function __construct()
    {
        $this->middleware(['auth']);
        // $this->middleware('can:adjust-stock')->only(['create','store','edit','update','destroy']);
    }

    /**
     * Adjustment List (with filters)
     */
    public function index(Request $request): View
    {
        $query = GSAdjustment::query()
            ->when($request->filled('company_id'), fn($q) =>
                $q->where('company_id', (int)$request->company_id)
            )
            ->when($request->filled('warehouse_id'), fn($q) =>
                $q->where('warehouse_id', (int)$request->warehouse_id)
            )
            ->when($request->filled('item_id'), fn($q) =>
                $q->where('item_id', (int)$request->item_id)
            )
            ->when($request->filled('date_from'), fn($q) =>
                $q->whereDate('adjusted_at', '>=', $request->date_from)
            )
            ->when($request->filled('date_to'), fn($q) =>
                $q->whereDate('adjusted_at', '<=', $request->date_to)
            )
            ->with(['company','warehouse'])   // eager load for list
            ->latest('adjusted_at')
            ->latest('id');

        $adjustments = $query->paginate(20)->withQueryString();

        // filters dropdown
        $companies  = Company::select('id','name')->orderBy('name')->get();
        $warehouses = Warehouse::select('id','name')->orderBy('name')->get();

        // view: resources/views/adjustment/gs_adjustment/index.blade.php
        return view('adjustment.gs_adjustment.index', compact('adjustments','companies','warehouses'));
    }

    /**
     * New Adjustment form
     */
    public function create(): View
    {
        $companies  = Company::select('id','name')->orderBy('name')->get();
        $warehouses = Warehouse::select('id','name')->orderBy('name')->get();

        // --- Dynamic column detection on products table ---
        $nameCol = Schema::hasColumn('products', 'name')
            ? 'name'
            : (Schema::hasColumn('products', 'product_name') ? 'product_name' : null);

        $unitCandidate = null;
        foreach (['unit_name','unit','uom_name','uom'] as $c) {
            if (Schema::hasColumn('products', $c)) { $unitCandidate = $c; break; }
        }

        $selects = ['id'];
        if ($nameCol) {
            $selects[] = $nameCol === 'name' ? 'name' : DB::raw("$nameCol as name");   // alias to name
        }
        if ($unitCandidate) {
            $selects[] = DB::raw("$unitCandidate as unit_name");                       // alias to unit_name
        }

        $orderBy = $nameCol ?: 'id';
        $items   = Product::select($selects)->orderBy($orderBy)->get();

        // attach stock fallback (0 if you don’t compute here)
        foreach ($items as $it) {
            if (!isset($it->stock)) $it->stock = 0;
        }

        // view: resources/views/adjustment/gs_adjustment/create.blade.php
        return view('adjustment.gs_adjustment.create', compact('companies','warehouses','items'));
    }

    /**
     * Store multi-line adjustment
     * Payload:
     * - company_id, warehouse_id, note
     * - items[n][item_id], items[n][qty], items[n][rate], items[n][adjust_type]
     */
    public function store(Request $request): RedirectResponse
    {
        // header validation
        $request->validate([
            'company_id'   => ['required','integer','min:1'],
            'warehouse_id' => ['required','integer','min:1'],
            'note'         => ['nullable','string','max:255'],
        ]);

        // lines
        $lines = $request->input('items', []);
        if (!is_array($lines) || count($lines) === 0) {
            return back()->withErrors(['items' => 'Please add at least one line.'])->withInput();
        }

        foreach ($lines as $i => $line) {
            $request->validate([
                "items.$i.item_id"     => ['required','integer','min:1'],
                "items.$i.qty"         => ['required','numeric','min:0.0001'],
                "items.$i.rate"        => ['nullable','numeric','min:0'],
                "items.$i.adjust_type" => ['required','in:IN,OUT'],
            ]);
        }

        $userId      = auth()->id();
        $nowAdjusted = now();

        DB::transaction(function () use ($request, $lines, $userId, $nowAdjusted) {
            foreach ($lines as $line) {
                $itemId = (int) ($line['item_id'] ?? 0);
                $qty    = (float) ($line['qty'] ?? 0);
                $rate   = (float) ($line['rate'] ?? 0);
                $type   = (string) ($line['adjust_type'] ?? 'OUT');

                // OUT => negative, IN => positive
                $diff = $type === 'IN' ? $qty : -$qty;

                GSAdjustment::create([
                    'company_id'   => (int) $request->input('company_id'),
                    'warehouse_id' => (int) $request->input('warehouse_id'),
                    'note'         => (string) $request->input('note', ''),

                    'item_id'      => $itemId,              // refers to products.id
                    'old_qty'      => 0,
                    'actual_qty'   => 0,
                    'diff_qty'     => round($diff, 3),
                    'reason'       => $type,                // IN/OUT (or use a dedicated column)
                    'adjusted_by'  => $userId,
                    'adjusted_at'  => $nowAdjusted,

                    // If your table lacks `rate`, remove this key or add the column in migration.
                    'rate'         => $rate,
                ]);

                // TODO: Update stock ledger/balance here if you maintain one
                // InventoryService::applyAdjustment($itemId, $diff, $request->input('warehouse_id'));
            }
        });

        return redirect()
            ->route('adjustment.new_gs.index')
            ->with('success', 'Stock adjusted successfully.');
    }

    /**
     * Edit (single row)
     */
    public function edit(int $id): View
    {
        $adjustment = GSAdjustment::findOrFail($id);

        $companies  = Company::select('id','name')->orderBy('name')->get();
        $warehouses = Warehouse::select('id','name')->orderBy('name')->get();

        // same dynamic product selects as create()
        $nameCol = Schema::hasColumn('products', 'name')
            ? 'name'
            : (Schema::hasColumn('products', 'product_name') ? 'product_name' : null);

        $unitCandidate = null;
        foreach (['unit_name','unit','uom_name','uom'] as $c) {
            if (Schema::hasColumn('products', $c)) { $unitCandidate = $c; break; }
        }

        $selects = ['id'];
        if ($nameCol) {
            $selects[] = $nameCol === 'name' ? 'name' : DB::raw("$nameCol as name");
        }
        if ($unitCandidate) {
            $selects[] = DB::raw("$unitCandidate as unit_name");
        }

        $orderBy = $nameCol ?: 'id';
        $items   = Product::select($selects)->orderBy($orderBy)->get();

        foreach ($items as $it) {
            if (!isset($it->stock)) $it->stock = 0;
        }

        // view: resources/views/adjustment/gs_adjustment/edit.blade.php (or reuse create)
        return view('adjustment.gs_adjustment.edit', compact('adjustment','companies','warehouses','items'));
    }

    /**
     * Update (single row)
     */
    public function update(Request $request, int $id): RedirectResponse
    {
        $adjustment = GSAdjustment::findOrFail($id);

        $validated = $request->validate([
            'company_id'   => ['required','integer','min:1'],
            'warehouse_id' => ['required','integer','min:1'],
            'note'         => ['nullable','string','max:255'],

            'item_id'      => ['required','integer','min:1'],
            'qty'          => ['required','numeric','min:0.0001'],
            'rate'         => ['nullable','numeric','min:0'],
            'adjust_type'  => ['required','in:IN,OUT'],
            'adjusted_at'  => ['nullable','date'],
        ]);

        $qty  = (float) $validated['qty'];
        $type = (string) $validated['adjust_type'];
        $diff = $type === 'IN' ? $qty : -$qty;

        $payload = [
            'company_id'   => (int) $validated['company_id'],
            'warehouse_id' => (int) $validated['warehouse_id'],
            'note'         => (string) ($validated['note'] ?? ''),

            'item_id'      => (int) $validated['item_id'],
            'old_qty'      => 0,
            'actual_qty'   => 0,
            'diff_qty'     => round($diff, 3),
            'reason'       => $type,
            'adjusted_by'  => auth()->id(),
            'adjusted_at'  => $validated['adjusted_at'] ?? now(),

            // Remove if your table has no `rate`
            'rate'         => (float) ($validated['rate'] ?? 0),
        ];

        DB::transaction(function () use ($adjustment, $payload) {
            // TODO: reverse previous ledger effect if needed
            $adjustment->update($payload);
        });

        return redirect()
            ->route('adjustment.new_gs.index')
            ->with('success', 'Adjustment updated.');
    }

    /**
     * Destroy
     */
    public function destroy(int $id): RedirectResponse
    {
        $adjustment = GSAdjustment::findOrFail($id);

        try {
            DB::transaction(function () use ($adjustment) {
                // TODO: reverse ledger effect if you applied in store()
                $adjustment->delete();
            });

            return redirect()
                ->route('adjustment.new_gs.index')
                ->with('success', 'Adjustment deleted.');
        } catch (\Throwable $e) {
            return redirect()
                ->route('adjustment.new_gs.index')
                ->with('error', 'Unable to delete: ' . $e->getMessage());
        }
    }
}

