<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Storage;
use Illuminate\Validation\Rule;

class FinishProductController extends Controller
{
    /** Master table for both raw & finish products */
    private string $table = 'rm_products';

    /** GET /products/finish-products */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $query = DB::table($this->table . ' as p')
            ->leftJoin('rm_categories as c', 'c.id', '=', 'p.category_id')
            ->leftJoin('rm_suppliers as s', 's.id', '=', 'p.supplier_id')
            ->leftJoin('warehouses as w', 'w.id', '=', 'p.warehouse_id')
            ->whereNull('p.deleted_at')
            ->where('p.type', 2) // 2 = finish
            ->select([
                'p.id', 'p.name', 'p.image', 'p.status',
                'p.opening_quantity', 'p.alert_quantity',
                'p.purchase_price', 'p.sale_price',
                'p.created_at', 'p.category_id',
                'c.name as category_name',
                's.name as supplier_name',
                'w.name as warehouse_name',
            ])
            ->orderBy('p.id', 'asc');

        if ($q !== '') {
            $query->where(function ($w) use ($q) {
                $w->where('p.name', 'like', "%{$q}%")
                  ->orWhere('c.name', 'like', "%{$q}%");
            });
        }

        $products = $query->paginate(12)->appends(['q' => $q]);

        return view('products.finish-products.index', compact('products', 'q'));
    }

    /** GET /products/finish-products/create */
    public function create()
    {
        $categories = DB::table('rm_categories')
            ->select('id','name')->orderBy('name')->get();
        $suppliers  = DB::table('rm_suppliers')
            ->select('id','name')->orderBy('name')->get();
        $warehouses = DB::table('warehouses')
            ->select('id','name')->orderBy('name')->get();

        return view('products.finish-products.create', compact('categories','suppliers','warehouses'));
    }

    /** POST /products/finish-products */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'              => ['required','string','max:191'],
            'category_id'       => ['required','exists:rm_categories,id'],
            'warehouse_id'      => ['nullable','exists:warehouses,id'],
            'supplier_id'       => ['nullable','exists:rm_suppliers,id'],
            'opening_quantity'  => ['nullable','numeric'],
            'alert_quantity'    => ['nullable','numeric'],
            'purchase_price'    => ['nullable','numeric'],
            'sale_price'        => ['nullable','numeric'],
            'vat'               => ['nullable','numeric'],
            'conversion_to_kg'  => ['nullable','numeric'],
            'status'            => ['required', Rule::in([0,1,'0','1'])],
            'description'       => ['nullable','string'],
            'image'             => ['nullable','image','max:2048'],
        ]);

        // Upload image
        $imagePath = null;
        if ($request->hasFile('image')) {
            $stored    = $request->file('image')->store('products', 'public'); // storage/app/public/products
            $imagePath = 'storage/' . $stored;                                  // public path
        }

        DB::table($this->table)->insert([
            'name'               => $data['name'],
            'category_id'        => $data['category_id'],
            'warehouse_id'       => $data['warehouse_id'] ?? null,
            'supplier_id'        => $data['supplier_id'] ?? null,
            'opening_quantity'   => $data['opening_quantity'] ?? 0,
            'alert_quantity'     => $data['alert_quantity']   ?? 0,
            'purchase_price'     => $data['purchase_price']   ?? null,
            'sale_price'         => $data['sale_price']       ?? null,
            'vat'                => $data['vat']              ?? null,
            'conversion_to_kg'   => $data['conversion_to_kg'] ?? null,
            'status'             => (int)$data['status'],
            'type'               => 2, // finish
            'image'              => $imagePath,
            'description'        => $data['description'] ?? null,
            'created_by'         => auth()->id(),
            'updated_by'         => auth()->id(),
            'created_at'         => now(),
            'updated_at'         => now(),
        ]);

        return redirect()
            ->route('products.finish-products.index')
            ->with('success', 'Finish product created successfully.');
    }

    /** GET /products/finish-products/{product}/edit */
    public function edit($id)
    {
        $product = DB::table($this->table)
            ->whereNull('deleted_at')
            ->where('type', 2) // finish
            ->where('id', $id)
            ->first();

        abort_if(!$product, 404);

        $categories = DB::table('rm_categories')->select('id','name')->orderBy('name')->get();
        $suppliers  = DB::table('rm_suppliers')->select('id','name')->orderBy('name')->get();
        $warehouses = DB::table('warehouses')->select('id','name')->orderBy('name')->get();

        return view('products.finish-products.edit', compact('product','categories','suppliers','warehouses'));
    }

    /** PUT /products/finish-products/{product} */
    public function update(Request $request, $id)
    {
        $product = DB::table($this->table)
            ->whereNull('deleted_at')
            ->where('type', 2)
            ->where('id', $id)
            ->first();

        abort_if(!$product, 404);

        $data = $request->validate([
            'name'              => ['required','string','max:191'],
            'category_id'       => ['required','exists:rm_categories,id'],
            'warehouse_id'      => ['nullable','exists:warehouses,id'],
            'supplier_id'       => ['nullable','exists:rm_suppliers,id'],
            'opening_quantity'  => ['nullable','numeric'],
            'alert_quantity'    => ['nullable','numeric'],
            'purchase_price'    => ['nullable','numeric'],
            'sale_price'        => ['nullable','numeric'],
            'vat'               => ['nullable','numeric'],
            'conversion_to_kg'  => ['nullable','numeric'],
            'status'            => ['required', Rule::in([0,1,'0','1'])],
            'description'       => ['nullable','string'],
            'image'             => ['nullable','image','max:2048'],
        ]);

        // Handle image replacement
        $imagePath = $product->image;
        if ($request->hasFile('image')) {
            if ($imagePath && str_starts_with($imagePath, 'storage/')) {
                $old = str_replace('storage/', '', $imagePath);
                Storage::disk('public')->delete($old);
            }
            $stored    = $request->file('image')->store('products', 'public');
            $imagePath = 'storage/' . $stored;
        }

        DB::table($this->table)
            ->where('id', $id)
            ->update([
                'name'               => $data['name'],
                'category_id'        => $data['category_id'],
                'warehouse_id'       => $data['warehouse_id'] ?? null,
                'supplier_id'        => $data['supplier_id'] ?? null,
                'opening_quantity'   => $data['opening_quantity'] ?? $product->opening_quantity,
                'alert_quantity'     => $data['alert_quantity']   ?? $product->alert_quantity,
                'purchase_price'     => $data['purchase_price']   ?? $product->purchase_price,
                'sale_price'         => $data['sale_price']       ?? $product->sale_price,
                'vat'                => $data['vat']              ?? $product->vat,
                'conversion_to_kg'   => $data['conversion_to_kg'] ?? $product->conversion_to_kg,
                'status'             => (int)$data['status'],
                'image'              => $imagePath,
                'description'        => $data['description'] ?? $product->description,
                'updated_by'         => auth()->id(),
                'updated_at'         => now(),
            ]);

        return redirect()
            ->route('products.finish-products.index')
            ->with('success', 'Finish product updated successfully.');
    }

    /** DELETE /products/finish-products/{product} */
    public function destroy($id)
    {
        $product = DB::table($this->table)
            ->whereNull('deleted_at')
            ->where('type', 2)
            ->where('id', $id)
            ->first();

        abort_if(!$product, 404);

        // delete image file (optional) — keeping file delete is common
        if ($product->image && str_starts_with($product->image, 'storage/')) {
            $old = str_replace('storage/', '', $product->image);
            Storage::disk('public')->delete($old);
        }

        // Soft delete (since rm_products has deleted_at)
        DB::table($this->table)->where('id', $id)->update([
            'deleted_at' => now(),
            'updated_at' => now(),
            'updated_by' => auth()->id(),
        ]);

        return redirect()
            ->route('products.finish-products.index')
            ->with('success', 'Finish product deleted successfully.');
    }
}
