<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\FinishProductAdjustment; // model with $table = 'fp_product_adjustments'
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;   // <-- for hasColumn checks

class FinishProductAdjustmentController extends Controller
{
    /**
     * List page.
     */
    public function index()
    {
        $adjustments = FinishProductAdjustment::with(['company', 'warehouse'])
            ->latest('id')
            ->paginate(15);

        return view('adjustment.finish_product.index', compact('adjustments'));
    }

    /**
     * Create form.
     */
    public function create()
    {
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        // Build product query safely (no 'type' column assumption)
        $productsQuery = Product::query()->where('status', 1);

        if (Schema::hasColumn('products', 'type')) {
            // Only apply when the column exists
            $productsQuery->where('type', 'finish_product');
        }

        // If your 'name' column might not exist, you can wrap with Schema::hasColumn too.
        $products = $productsQuery->orderBy('name')->get();

        return view('adjustment.finish_product.create', compact('companies', 'warehouses', 'products'));
    }

    /**
     * Store master row (aggregate numbers).
     * Extend to save per-item rows if you have an items table.
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'company_id'                     => ['required', 'exists:companies,id'],
            'warehouse_id'                   => ['required', 'exists:warehouses,id'],
            'date'                           => ['required', 'date'],
            'note'                           => ['nullable', 'string', 'max:1000'],

            'items'                          => ['required', 'array', 'min:1'],
            'items.*.product_id'             => ['required', 'exists:products,id'],
            'items.*.qty'                    => ['required', 'numeric', 'min:0.01'],
            'items.*.price'                  => ['required', 'numeric', 'min:0'],
            'items.*.adjust_type'            => ['required', 'in:IN,OUT,in,out'],
        ]);

        // Aggregate totals
        $totalQty   = 0;
        $totalValue = 0;

        foreach ($data['items'] as $row) {
            $qty   = (float) $row['qty'];
            $price = (float) $row['price'];
            $totalQty   += $qty;
            $totalValue += ($qty * $price);
        }

        DB::transaction(function () use ($request, $totalQty, $totalValue) {
            // Save master
            FinishProductAdjustment::create([
                'company_id'   => $request->company_id,
                'warehouse_id' => $request->warehouse_id,
                'date'         => $request->date,
                'total_qty'    => $totalQty,
                'total_value'  => $totalValue,
                'note'         => $request->note,
            ]);

            // If/when you add an items table, save each row here, e.g.:
            // foreach ($request->items as $row) {
            //     FinishProductAdjustmentItem::create([
            //         'finish_product_adjustment_id' => $master->id,
            //         'product_id'   => $row['product_id'],
            //         'qty'          => $row['qty'],
            //         'price'        => $row['price'],
            //         'subtotal'     => $row['qty'] * $row['price'],
            //         'adjust_type'  => strtoupper($row['adjust_type']),
            //     ]);
            // }
        });

        return redirect()
            ->route('adjustment.finish_product.index')
            ->with('success', 'Finish product adjustment created.');
    }

    /**
     * Show page (view/print).
     */
    public function show($id)
    {
        $adjustment = FinishProductAdjustment::with(['company', 'warehouse'])
            ->findOrFail($id);

        return view('adjustment.finish_product.show', compact('adjustment'));
    }

    /**
     * Edit form.
     * (Extend to load items if you add an items table.)
     */
    public function edit($id)
    {
        $adjustment = FinishProductAdjustment::findOrFail($id);
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        $productsQuery = Product::query()->where('status', 1);
        if (Schema::hasColumn('products', 'type')) {
            $productsQuery->where('type', 'finish_product');
        }
        $products = $productsQuery->orderBy('name')->get();

        return view('adjustment.finish_product.edit', compact('adjustment', 'companies', 'warehouses', 'products'));
    }

    /**
     * Update master info. Recalculate totals if you also update items.
     */
    public function update(Request $request, $id)
    {
        $adjustment = FinishProductAdjustment::findOrFail($id);

        $data = $request->validate([
            'company_id'   => ['required', 'exists:companies,id'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'date'         => ['required', 'date'],
            'note'         => ['nullable', 'string', 'max:1000'],
        ]);

        $adjustment->update([
            'company_id'   => $data['company_id'],
            'warehouse_id' => $data['warehouse_id'],
            'date'         => $data['date'],
            'note'         => $data['note'] ?? null,
            // total_qty/total_value: recompute if editing items
        ]);

        return redirect()
            ->route('adjustment.finish_product.index')
            ->with('success', 'Finish product adjustment updated.');
    }

    /**
     * Delete master (and items if you add them).
     */
    public function destroy($id)
    {
        $adjustment = FinishProductAdjustment::findOrFail($id);

        DB::transaction(function () use ($adjustment) {
            // FinishProductAdjustmentItem::where('finish_product_adjustment_id', $adjustment->id)->delete();
            $adjustment->delete();
        });

        return redirect()
            ->route('adjustment.finish_product.index')
            ->with('success', 'Finish product adjustment deleted.');
    }
}

