<?php

namespace App\Http\Controllers;

use App\Models\FinishBag;
use App\Models\Warehouse;
use Illuminate\Http\Request;

class FinishBagController extends Controller
{
    /**
     * GET /finish-bags
     */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $bags = FinishBag::with('warehouse')
            ->when($q, function ($qr) use ($q) {
                $qr->where('name', 'like', "%{$q}%")
                   ->orWhereHas('warehouse', fn ($w) => $w->where('name', 'like', "%{$q}%"));
            })
            ->orderBy('id', 'asc')
            ->paginate(15)
            ->withQueryString();

        // ✅ fixed: products.finish-bags.index (not 'produtcs')
        return view('products.finish-bags.index', compact('bags', 'q'));
    }

    /**
     * GET /finish-bags/create
     */
    public function create()
    {
        $warehouses = Warehouse::orderBy('name')->pluck('name', 'id'); // [id => name]

        return view('products.finish-bags.create', compact('warehouses'));
    }

    /**
     * POST /finish-bags
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'         => ['required', 'string', 'max:191'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'opening_qty'  => ['nullable', 'numeric'],
            'alert_qty'    => ['nullable', 'numeric'],
            'bag_price'    => ['nullable', 'numeric'],
            'status'       => ['required', 'in:active,inactive,1,0'],
        ]);

        // normalize status to 1/0
        $status = in_array($data['status'], ['1', 1, true, 'active'], true) ? 1 : 0;

        FinishBag::create([
            'name'         => $data['name'],
            'warehouse_id' => $data['warehouse_id'],
            'opening_qty'  => $data['opening_qty'] ?? 0,
            'alert_qty'    => $data['alert_qty']   ?? 0,
            'bag_price'    => $data['bag_price']   ?? 0,
            'status'       => $status,
            'created_by'   => auth()->id(),
            'updated_by'   => auth()->id(),
        ]);

        return redirect()
            ->route('products.finish-bags.index')
            ->with('success', 'Finish bag created successfully.');
    }

    /**
     * GET /finish-bags/{bag}/edit
     * Route-model binding expects route param {bag}
     */
    public function edit(FinishBag $bag)
    {
        $warehouses = Warehouse::orderBy('name')->pluck('name', 'id');

        // ✅ fixed: products.finish-bags.edit (not 'bags.finish-bags.edit')
        return view('products.finish-bags.edit', compact('bag', 'warehouses'));
    }

    /**
     * PUT/PATCH /finish-bags/{bag}
     */
    public function update(Request $request, FinishBag $bag)
    {
        $data = $request->validate([
            'name'         => ['required', 'string', 'max:191'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'opening_qty'  => ['nullable', 'numeric'],
            'alert_qty'    => ['nullable', 'numeric'],
            'bag_price'    => ['nullable', 'numeric'],
            'status'       => ['required', 'in:active,inactive,1,0'],
        ]);

        $status = in_array($data['status'], ['1', 1, true, 'active'], true) ? 1 : 0;

        $bag->update([
            'name'         => $data['name'],
            'warehouse_id' => $data['warehouse_id'],
            'opening_qty'  => $data['opening_qty'] ?? $bag->opening_qty,
            'alert_qty'    => $data['alert_qty']   ?? $bag->alert_qty,
            'bag_price'    => $data['bag_price']   ?? $bag->bag_price,
            'status'       => $status,
            'updated_by'   => auth()->id(),
        ]);

        return redirect()
            ->route('products.finish-bags.index')
            ->with('success', 'Finish bag updated successfully.');
    }

    /**
     * DELETE /finish-bags/{bag}
     */
    public function destroy(FinishBag $bag)
    {
        $bag->delete();

        return redirect()
            ->route('products.finish-bags.index')
            ->with('success', 'Finish bag deleted successfully.');
    }
}
