<?php

namespace App\Http\Controllers;

use App\Models\Company;
use App\Models\Product;
use App\Models\Warehouse;
use App\Models\FinishBagAdjustment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

class FinishBagAdjustmentController extends Controller
{
    /**
     * List with optional filters.
     */
    public function index(Request $request)
    {
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::orderBy('name')->get();

        $q = FinishBagAdjustment::with(['company', 'warehouse'])->latest('id');

        if ($request->filled('company_id')) {
            $q->where('company_id', $request->company_id);
        }
        if ($request->filled('warehouse_id')) {
            $q->where('warehouse_id', $request->warehouse_id);
        }
        if ($request->filled('from')) {
            $q->whereDate('date', '>=', $request->from);
        }
        if ($request->filled('to')) {
            $q->whereDate('date', '<=', $request->to);
        }

        $adjustments = $q->paginate(15);

        return view('adjustment.finish_bag.index', compact('adjustments', 'companies', 'warehouses'));
    }

    /**
     * Create form.
     */
    public function create()
    {
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        // Load “bags” safely; do not assume a 'type' column exists.
        $productsQuery = Product::query()->where('status', 1);
        if (Schema::hasColumn('products', 'type')) {
            // If you tag bags in the products table, adapt the value if needed
            $productsQuery->where('type', 'finish_bag');
        }
        $products = $productsQuery->orderBy(
            Schema::hasColumn('products', 'name') ? 'name' : 'id'
        )->get();

        return view('adjustment.finish_bag.create', compact('companies', 'warehouses', 'products'));
    }

    /**
     * Store master (and later detail rows if you add an items table).
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'company_id'                     => ['required', 'exists:companies,id'],
            'warehouse_id'                   => ['required', 'exists:warehouses,id'],
            'date'                           => ['required', 'date'],
            'note'                           => ['nullable', 'string', 'max:1000'],

            'items'                          => ['required', 'array', 'min:1'],
            'items.*.product_id'             => ['required', 'exists:products,id'],
            'items.*.qty'                    => ['required', 'numeric', 'min:0.01'],
            'items.*.price'                  => ['nullable', 'numeric', 'min:0'],
            'items.*.adjust_type'            => ['required', 'in:IN,OUT,in,out'],
        ]);

        $totalQty   = 0;
        $totalValue = 0;

        foreach ($data['items'] as $row) {
            $qty   = (float) $row['qty'];
            $price = (float) ($row['price'] ?? 0);
            $totalQty   += $qty;
            $totalValue += $qty * $price;
        }

        DB::transaction(function () use ($request, $totalQty, $totalValue) {
            $master = FinishBagAdjustment::create([
                'company_id'   => $request->company_id,
                'warehouse_id' => $request->warehouse_id,
                'date'         => $request->date,
                'total_qty'    => $totalQty,
                'total_value'  => $totalValue,
                'note'         => $request->note,
            ]);

            // If you introduce an items table later, save each item here.
            // foreach ($request->items as $row) {
            //     FinishBagAdjustmentItem::create([
            //         'finish_bag_adjustment_id' => $master->id,
            //         'product_id'   => $row['product_id'],
            //         'qty'          => $row['qty'],
            //         'price'        => $row['price'] ?? 0,
            //         'subtotal'     => ((float)$row['qty']) * ((float)($row['price'] ?? 0)),
            //         'adjust_type'  => strtoupper($row['adjust_type']),
            //     ]);
            // }
        });

        return redirect()
            ->route('adjustment.finish_bag.index')
            ->with('success', 'Finish bag adjustment created.');
    }

    /**
     * Show page (view/print).
     */
    public function show($id)
    {
        $adjustment = FinishBagAdjustment::with(['company', 'warehouse'])->findOrFail($id);
        return view('adjustment.finish_bag.show', compact('adjustment'));
    }

    /**
     * Edit form.
     */
    public function edit($id)
    {
        $adjustment = FinishBagAdjustment::findOrFail($id);
        $companies  = Company::orderBy('name')->get();
        $warehouses = Warehouse::where('status', 1)->orderBy('name')->get();

        $productsQuery = Product::query()->where('status', 1);
        if (Schema::hasColumn('products', 'type')) {
            $productsQuery->where('type', 'finish_bag');
        }
        $products = $productsQuery->orderBy(
            Schema::hasColumn('products', 'name') ? 'name' : 'id'
        )->get();

        return view('adjustment.finish_bag.edit', compact('adjustment', 'companies', 'warehouses', 'products'));
    }

    /**
     * Update master (recompute totals if you also edit items).
     */
    public function update(Request $request, $id)
    {
        $adjustment = FinishBagAdjustment::findOrFail($id);

        $data = $request->validate([
            'company_id'   => ['required', 'exists:companies,id'],
            'warehouse_id' => ['required', 'exists:warehouses,id'],
            'date'         => ['required', 'date'],
            'note'         => ['nullable', 'string', 'max:1000'],
        ]);

        $adjustment->update([
            'company_id'   => $data['company_id'],
            'warehouse_id' => $data['warehouse_id'],
            'date'         => $data['date'],
            'note'         => $data['note'] ?? null,
        ]);

        return redirect()
            ->route('adjustment.finish_bag.index')
            ->with('success', 'Finish bag adjustment updated.');
    }

    /**
     * Delete master (and detail rows if present).
     */
    public function destroy($id)
    {
        $adjustment = FinishBagAdjustment::findOrFail($id);

        DB::transaction(function () use ($adjustment) {
            // FinishBagAdjustmentItem::where('finish_bag_adjustment_id', $adjustment->id)->delete();
            $adjustment->delete();
        });

        return redirect()
            ->route('adjustment.finish_bag.index')
            ->with('success', 'Finish bag adjustment deleted.');
    }
}

