<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class EmployeeController extends Controller
{
    /* ==================== EMPLOYEES ==================== */

    /** GET /employee  => employee.index */
    public function index(Request $request)
    {
        // Filters
        $designationId = $request->input('designation_id');
        $empId         = $request->input('emp_id');

        // Dropdowns
        $designations = DB::table('rm_employee_designations')
            ->select('id','name')->orderBy('name')->get();

        $employeesForFilter = DB::table('rm_employees')
            ->select('id','name')->orderBy('name')->get();

        // List with joins (users join added for tooltip)
        $rows = DB::table('rm_employees as e')
            ->leftJoin('rm_employee_designations as dg', 'dg.id', '=', 'e.designation_id')
            ->leftJoin('rm_employee_departments  as dp', 'dp.id', '=', 'e.department_id')
            ->leftJoin('users as cu', 'cu.id', '=', 'e.created_by') // creator
            ->leftJoin('users as uu', 'uu.id', '=', 'e.updated_by') // updater
            ->when($designationId, fn($q) => $q->where('e.designation_id', $designationId))
            ->when($empId, fn($q) => $q->where('e.id', $empId))
            ->select([
                'e.id','e.name','e.phone','e.date_of_birth','e.department_id','e.designation_id',
                'e.address','e.nid','e.gender','e.religion','e.nationality','e.image',
                'e.created_at','e.updated_at',
                'dg.name as designation_name','dp.name as department_name',
                DB::raw("COALESCE(cu.name, cu.email) as created_by_name"),
                DB::raw("COALESCE(uu.name, uu.email) as updated_by_name"),
            ])
            ->latest('e.id')
            ->paginate(20)
            ->appends($request->query());

        return view('employee.index', [
            'rows'         => $rows,
            'designations' => $designations,
            'employees'    => $employeesForFilter,
        ]);
    }

    /** GET /employee/create => employee.create */
    public function create()
    {
        $designations = DB::table('rm_employee_designations')->select('id','name')->orderBy('name')->get();
        $departments  = DB::table('rm_employee_departments')->select('id','name')->orderBy('name')->get();

        return view('employee.create', compact('designations','departments'));
    }

    /** POST /employee => employee.store */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'           => ['required','string','max:191'],
            'phone'          => ['required','string','max:50'],
            'dob'            => ['nullable','date'],
            'designation_id' => ['required','integer'],
            'department_id'  => ['required','integer'],
            'address'        => ['required','string','max:255'],
            'nid'            => ['nullable','string','max:100'],
            'gender'         => ['nullable','in:Male,Female,Other'],
            'religion'       => ['nullable','string','max:100'],
            'nationality'    => ['nullable','string','max:100'],
            'image'          => ['nullable','image','mimes:jpg,jpeg,png,webp','max:2048'],
        ]);

        // logged-in user (null হলে null-ই সেভ হবে)
        $userId = auth()->id();

        // file upload
        $imagePath = null;
        if ($request->hasFile('image')) {
            // php artisan storage:link করা থাকতে হবে
            $imagePath = $request->file('image')->store('employees', 'public');
        }

        $insert = [
            'name'           => $data['name'],
            'phone'          => $data['phone'],
            'date_of_birth'  => !empty($data['dob']) ? Carbon::parse($data['dob'])->format('Y-m-d') : null,
            'designation_id' => $data['designation_id'],
            'department_id'  => $data['department_id'],
            'address'        => $data['address'],
            'nid'            => $data['nid'] ?? null,
            'gender'         => $data['gender'] ?? 'Male',
            'religion'       => $data['religion'] ?? null,
            'nationality'    => $data['nationality'] ?? null,
            'image'          => $imagePath,
            'created_by'     => $userId,
            'updated_by'     => $userId,
            'created_at'     => now(),
            'updated_at'     => now(),
        ];

        DB::table('rm_employees')->insert($insert);

        return redirect()->route('employee.index')->with('success', 'Employee saved successfully.');
    }

    /* ==================== DESIGNATIONS ==================== */

    public function designationsIndex()
    {
        $rows = DB::table('rm_employee_designations')->orderBy('id')->get();

        return view('employee.designations.index', [
            'designations' => $rows,
            'rows'         => $rows,
        ]);
    }

    public function designationStore(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:191'],
        ]);

        $userId = auth()->id();

        DB::table('rm_employee_designations')->insert([
            'name'       => $data['name'],
            'created_by' => $userId,
            'updated_by' => $userId,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return redirect()->route('employee.designations')->with('success', 'Designation saved.');
    }

    /* ==================== DEPARTMENTS ==================== */

    public function departmentsIndex()
    {
        $rows = DB::table('rm_employee_departments')->orderBy('id')->get();

        return view('employee.departments.index', [
            'departments' => $rows,
            'rows'        => $rows,
        ]);
    }

    public function departmentStore(Request $request)
    {
        $data = $request->validate([
            'name' => ['required','string','max:191'],
        ]);

        $userId = auth()->id();

        DB::table('rm_employee_departments')->insert([
            'name'       => $data['name'],
            'created_by' => $userId,
            'updated_by' => $userId,
            'created_at' => now(),
            'updated_at' => now(),
        ]);

        return redirect()->route('employee.departments')->with('success', 'Department saved.');
    }
}
