<?php

namespace App\Http\Controllers;

use App\Models\Dayar;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class DayarController extends Controller
{
    /** GET /products/dayars */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $dayars = Dayar::with(['createdBy', 'updatedBy'])
            ->when($q, fn($qr) => $qr->where('name', 'like', "%{$q}%"))
            ->orderBy('id', 'asc')
            ->paginate(12)
            ->withQueryString();

        return view('products.dayars.index', compact('dayars', 'q'));
    }

    /** GET /products/dayars/create */
    public function create()
    {
        return view('products.dayars.create');
    }

    /** POST /products/dayars */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'   => ['required', 'string', 'max:191', 'unique:dayars,name'],
            'status' => ['required', Rule::in(['0','1'])],
        ]);

        Dayar::create([
            'name'      => $data['name'],
            'status'    => (int)$data['status'],
            'created_by'=> auth()->id() ?? null,
            'updated_by'=> auth()->id() ?? null,
        ]);

        return redirect()->route('products.dayars.index')
            ->with('success', 'Dayar created successfully.');
    }

    /** GET /products/dayars/{dayar}/edit */
    public function edit(Dayar $dayar)
    {
        return view('products.dayars.edit', compact('dayar'));
    }

    /** PUT /products/dayars/{dayar} */
    public function update(Request $request, Dayar $dayar)
    {
        $data = $request->validate([
            'name'   => ['required', 'string', 'max:191', Rule::unique('dayars', 'name')->ignore($dayar->id)],
            'status' => ['required', Rule::in(['0','1'])],
        ]);

        $dayar->update([
            'name'      => $data['name'],
            'status'    => (int)$data['status'],
            'updated_by'=> auth()->id() ?? $dayar->updated_by,
        ]);

        return redirect()->route('products.dayars.index')
            ->with('success', 'Dayar updated successfully.');
    }

    /** DELETE /products/dayars/{dayar} */
    public function destroy(Dayar $dayar)
    {
        $dayar->delete();

        return redirect()->route('products.dayars.index')
            ->with('success', 'Dayar deleted successfully.');
    }
}

