<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Session;
use Symfony\Component\HttpFoundation\StreamedResponse;

class CustomerCsvController extends Controller
{
    private const SESSION_KEY = 'csv_customers_rows';

    public function uploadForm()
    {
        return view('party.customers.csv_upload');
    }

    public function uploadStore(Request $request)
    {
        $request->validate([
            'csv_file' => ['required','file','mimes:csv,txt','max:5120'],
        ]);

        $path = $request->file('csv_file')->getRealPath();
        $handle = fopen($path, 'r');

        if (!$handle) {
            return back()->withErrors(['csv_file' => 'Could not open the uploaded file.']);
        }

        // Read header
        $header = fgetcsv($handle);
        if (!$header) {
            fclose($handle);
            return back()->withErrors(['csv_file' => 'Empty CSV or invalid format.']);
        }

        // Normalize header keys
        $map = [];
        foreach ($header as $i => $h) {
            $k = strtolower(trim($h));
            $k = str_replace([' ', '-', '__'], ['_', '_', '_'], $k);
            $map[$i] = $k;
        }

        $rows = [];
        while (($data = fgetcsv($handle)) !== false) {
            if (count(array_filter($data, fn($x)=>$x!==null && $x!==''))
                === 0) continue; // skip blank rows

            $row = [
                'name'             => '',
                'company_name'     => '',
                'email'            => '',
                'address'          => '',
                'personal_phone'   => '',
                'executive'        => '',
                'area'             => '',
                'opening_balance'  => 0,
                'due_limit'        => 0,
                'remark'           => '',
                'party_type'       => '',
                'status'           => 1,
            ];

            foreach ($data as $i => $val) {
                $key = $map[$i] ?? null;
                if (!$key) continue;
                $val = trim($val);

                // allow several header aliases
                switch ($key) {
                    case 'customer_name':
                    case 'name': $row['name'] = $val; break;

                    case 'company':
                    case 'company_name': $row['company_name'] = $val; break;

                    case 'phone':
                    case 'mobile':
                    case 'personal_phone': $row['personal_phone'] = $val; break;

                    case 'opening':
                    case 'opening_balance': $row['opening_balance'] = (float)$val; break;

                    case 'customer_due_limit':
                    case 'due_limit': $row['due_limit'] = (float)$val; break;

                    case 'remarks':
                    case 'remark': $row['remark'] = $val; break;

                    default:
                        if (array_key_exists($key, $row)) {
                            $row[$key] = $key === 'status' ? (int)($val ? 1:0) : $val;
                        }
                }
            }
            $rows[] = $row;
        }
        fclose($handle);

        Session::put(self::SESSION_KEY, array_values($rows));

        return redirect()->route('party.customers.csv')
            ->with('success','File Uploaded Successfully');
    }

    public function index()
    {
        $rows = Session::get(self::SESSION_KEY, []);
        return view('party.customers.csv_list', ['rows'=>$rows]);
    }

    public function update(Request $request, $key)
    {
        $rows = Session::get(self::SESSION_KEY, []);
        if (!isset($rows[$key])) return back()->withErrors('Row not found');

        $rows[$key]['name']            = $request->input('name', $rows[$key]['name']);
        $rows[$key]['company_name']    = $request->input('company_name', $rows[$key]['company_name']);
        $rows[$key]['email']           = $request->input('email', $rows[$key]['email']);
        $rows[$key]['address']         = $request->input('address', $rows[$key]['address']);
        $rows[$key]['personal_phone']  = $request->input('personal_phone', $rows[$key]['personal_phone']);
        $rows[$key]['executive']       = $request->input('executive', $rows[$key]['executive']);
        $rows[$key]['area']            = $request->input('area', $rows[$key]['area']);
        $rows[$key]['opening_balance'] = (float)$request->input('opening_balance', $rows[$key]['opening_balance']);
        $rows[$key]['due_limit']       = (float)$request->input('due_limit', $rows[$key]['due_limit']);
        $rows[$key]['remark']          = $request->input('remark', $rows[$key]['remark']);
        $rows[$key]['party_type']      = $request->input('party_type', $rows[$key]['party_type']);
        $rows[$key]['status']          = (int)$request->boolean('status', $rows[$key]['status']);

        Session::put(self::SESSION_KEY, $rows);
        return back()->with('success','Row updated');
    }

    public function destroy($key)
    {
        $rows = Session::get(self::SESSION_KEY, []);
        if (isset($rows[$key])) {
            unset($rows[$key]);
            $rows = array_values($rows);
            Session::put(self::SESSION_KEY, $rows);
        }
        return back()->with('success','Row deleted');
    }

    public function destroyAll()
    {
        Session::forget(self::SESSION_KEY);
        return back()->with('success','All rows deleted');
    }

    public function importFirst50()
    {
        // Demo only: এখানে আপনার Customer মডেলে ইনসার্ট করবেন
        // এখন শুধু মেসেজ দেখাচ্ছি:
        return back()->with('success','Imported first 50 customers (demo)');
    }

    public function downloadSample(): StreamedResponse
    {
        $csv = implode("\n", [
            "Customer Name,Company Name,Email,Address,Phone Personal,Executive,Area,Opening Balance,Customer Due Limit,Remark,Party Type,Status",
            "Khan,Core4 Software Ltd,akashsd1@gmail.com,Dhaka,1813695989,Sarowar Hossen,Narayanganj Zone,0,0,Good,Due,1",
        ]);

        return response()->streamDownload(function() use ($csv){
            echo $csv;
        }, 'customers_sample.csv', [
            'Content-Type' => 'text/csv; charset=UTF-8',
        ]);
    }
}
