<?php

namespace App\Http\Controllers;

use App\Models\BagType;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class BagTypeController extends Controller
{
    /**
     * GET /products/bag-types
     * Route: products.bag-types.index
     */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $bagTypes = BagType::query()
            ->when($q, function ($qr) use ($q) {
                $qr->where(function ($w) use ($q) {
                    $w->where('name', 'like', "%{$q}%")
                      ->orWhere('bag_size', 'like', "%{$q}%"); // ← correct column
                });
            })
            ->orderByDesc('id')
            ->paginate(20)
            ->withQueryString();

        return view('products.bag-types.index', compact('bagTypes', 'q'));
    }

    /**
     * GET /products/bag-types/create
     * Route: products.bag-types.create
     */
    public function create()
    {
        return view('products.bag-types.create');
    }

    /**
     * POST /products/bag-types
     * Route: products.bag-types.store
     */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'     => ['required', 'string', 'max:191', 'unique:rm_bag_types,name'],
            'bag_size' => ['required', 'numeric', 'min:0'],                 // ← validate bag_size
            'status'   => ['required', Rule::in(['0','1',0,1,true,false])],
        ]);

        BagType::create([
            'name'       => $data['name'],
            'bag_size'   => $data['bag_size'],                               // ← save bag_size
            'status'     => (string)((int)(bool)$data['status']),            // normalise to '1'/'0'
            'created_by' => auth()->id() ?? 1,
        ]);

        return redirect()
            ->route('products.bag-types.index')
            ->with('success', 'Bag type created successfully.');
    }

    /**
     * GET /products/bag-types/{bag}/edit
     * Route param kept as {bag}
     * Route: products.bag-types.edit
     */
    public function edit(BagType $bag)
    {
        return view('products.bag-types.edit', ['bagType' => $bag]);
    }

    /**
     * PUT/PATCH /products/bag-types/{bag}
     * Route: products.bag-types.update
     */
    public function update(Request $request, BagType $bag)
    {
        $data = $request->validate([
            'name'     => [
                'required', 'string', 'max:191',
                Rule::unique('rm_bag_types', 'name')->ignore($bag->id),
            ],
            'bag_size' => ['required', 'numeric', 'min:0'],                  // ← validate bag_size
            'status'   => ['required', Rule::in(['0','1',0,1,true,false])],
        ]);

        $bag->update([
            'name'       => $data['name'],
            'bag_size'   => $data['bag_size'],                               // ← update bag_size
            'status'     => (string)((int)(bool)$data['status']),
            'updated_by' => auth()->id() ?? null,
        ]);

        return redirect()
            ->route('products.bag-types.index')
            ->with('success', 'Bag type updated successfully.');
    }

    /**
     * DELETE /products/bag-types/{bag}
     * Route: products.bag-types.destroy
     */
    public function destroy(BagType $bag)
    {
        $bag->delete();

        return redirect()
            ->route('products.bag-types.index')
            ->with('success', 'Bag type deleted successfully.');
    }
}

