<?php

namespace App\Http\Controllers;

use App\Models\BagExchange;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Schema;

// These should exist in your app; adjust namespaces if different
use App\Models\Warehouse;
use App\Models\Product;

class BagExchangeController extends Controller
{
    /* =========================
     *         INDEX
     * ========================= */
    public function index(Request $request)
    {
        $q       = trim((string) $request->get('q', ''));
        $perPage = (int) $request->get('per_page', 15);

        $rows = BagExchange::query()
            ->when($q !== '', function ($query) use ($q) {
                $table = (new BagExchange)->getTable();
                $query->where(function ($qb) use ($q, $table) {
                    foreach (['remarks', 'reference', 'type'] as $col) {
                        if (Schema::hasColumn($table, $col)) {
                            $qb->orWhere($col, 'like', "%{$q}%");
                        }
                    }
                });
            })
            ->latest('id')
            ->paginate($perPage)
            ->withQueryString();

        return view('bag_exchange.index', compact('rows', 'q', 'perPage'));
    }

    /* =========================
     *         CREATE
     * ========================= */
    public function create()
    {
        $warehouses    = class_exists(Warehouse::class)
            ? Warehouse::query()->orderBy('name')->get(['id','name'])
            : collect();

        // If you don’t have a separate bag-warehouse table, reuse warehouses.
        $bagWarehouses = $warehouses;

        $products      = class_exists(Product::class)
            ? Product::query()->orderBy('name')->get(['id','name'])
            : collect();

        // If there is a FinishBag model, use it; otherwise fall back to products list.
        if (class_exists(\App\Models\FinishBag::class)) {
            $finishBags = \App\Models\FinishBag::query()->orderBy('name')->get(['id','name']);
        } else {
            $finishBags = $products;
        }

        return view('bag_exchange.create', compact(
            'warehouses', 'bagWarehouses', 'products', 'finishBags'
        ));
    }

    /* =========================
     *         STORE
     * ========================= */
    public function store(Request $request)
    {
        // Fields match your create blade
        $data = $request->validate([
            'warehouse_id'      => ['required','integer'],
            'date'              => ['required','date'],
            'bag_warehouse_id'  => ['required','integer'],
            'ex_warehouse_id'   => ['required','integer','different:warehouse_id'],
            'product_id'        => ['required','integer'],
            'ex_product_id'     => ['required','integer'],
            'finish_bag_id'     => ['required','integer'],
            'ex_finish_bag_id'  => ['required','integer'],
            'product_stock_kg'  => ['nullable','numeric'],
            'total_kg'          => ['required','numeric','min:0'],
            'status'            => ['nullable','in:0,1'],
        ]);

        $table = (new BagExchange)->getTable();

        if (Schema::hasColumn($table, 'status') && !array_key_exists('status', $data)) {
            $data['status'] = 1;
        }
        if (Schema::hasColumn($table, 'created_by')) {
            $data['created_by'] = Auth::id();
        }
        if (Schema::hasColumn($table, 'updated_by')) {
            $data['updated_by'] = Auth::id();
        }

        BagExchange::create($data);

        return redirect()
            ->route('bag_exchange.index')
            ->with('success', 'Bag exchange created successfully.');
    }

    /* =========================
     *         SHOW (optional)
     * ========================= */
    public function show(BagExchange $bagExchange)
    {
        return view('bag_exchange.show', ['exchange' => $bagExchange]);
    }

    /* =========================
     *         EDIT
     * ========================= */
    public function edit(BagExchange $bagExchange)
    {
        // Same lists as create so the dropdowns are populated
        $warehouses    = class_exists(Warehouse::class)
            ? Warehouse::query()->orderBy('name')->get(['id','name'])
            : collect();

        $bagWarehouses = $warehouses;

        $products      = class_exists(Product::class)
            ? Product::query()->orderBy('name')->get(['id','name'])
            : collect();

        if (class_exists(\App\Models\FinishBag::class)) {
            $finishBags = \App\Models\FinishBag::query()->orderBy('name')->get(['id','name']);
        } else {
            $finishBags = $products;
        }

        return view('bag_exchange.edit', [
            'exchange'       => $bagExchange,
            'warehouses'     => $warehouses,
            'bagWarehouses'  => $bagWarehouses,
            'products'       => $products,
            'finishBags'     => $finishBags,
        ]);
    }

    /* =========================
     *         UPDATE
     * ========================= */
    public function update(Request $request, BagExchange $bagExchange)
    {
        $data = $request->validate([
            'warehouse_id'      => ['required','integer'],
            'date'              => ['required','date'],
            'bag_warehouse_id'  => ['required','integer'],
            'ex_warehouse_id'   => ['required','integer','different:warehouse_id'],
            'product_id'        => ['required','integer'],
            'ex_product_id'     => ['required','integer'],
            'finish_bag_id'     => ['required','integer'],
            'ex_finish_bag_id'  => ['required','integer'],
            'product_stock_kg'  => ['nullable','numeric'],
            'total_kg'          => ['required','numeric','min:0'],
            'status'            => ['nullable','in:0,1'],
        ]);

        if (Schema::hasColumn($bagExchange->getTable(), 'updated_by')) {
            $data['updated_by'] = Auth::id();
        }

        $bagExchange->update($data);

        return redirect()
            ->route('bag_exchange.index')
            ->with('success', 'Bag exchange updated successfully.');
    }

    /* =========================
     *         DESTROY
     * ========================= */
    public function destroy(BagExchange $bagExchange)
    {
        $bagExchange->delete();

        return redirect()
            ->route('bag_exchange.index')
            ->with('success', 'Bag exchange deleted successfully.');
    }
}

