<?php

namespace App\Http\Controllers;

use App\Models\Area;
use Illuminate\Http\Request;
use Illuminate\Validation\Rule;

class AreaController extends Controller
{
    /** GET /products/areas */
    public function index(Request $request)
    {
        $q = trim($request->get('q', ''));

        $areas = Area::with(['createdBy', 'updatedBy'])
            ->when($q, fn($qr) => $qr->where('name', 'like', "%{$q}%"))
            ->orderBy('id', 'asc')
            ->paginate(12)
            ->withQueryString();

        return view('products.areas.index', compact('areas', 'q'));
    }

    /** GET /products/areas/create */
    public function create()
    {
        return view('products.areas.create');
    }

    /** POST /products/areas */
    public function store(Request $request)
    {
        $data = $request->validate([
            'name'   => ['required', 'string', 'max:191', 'unique:areas,name'],
            'status' => ['required', Rule::in(['0','1'])],
        ]);

        Area::create([
            'name'       => $data['name'],
            'status'     => (int)$data['status'],
            'created_by' => auth()->id() ?? null,
            'updated_by' => auth()->id() ?? null,
        ]);

        return redirect()->route('products.areas.index')
            ->with('success', 'Area created successfully.');
    }

    /** GET /products/areas/{area}/edit */
    public function edit(Area $area)
    {
        return view('products.areas.edit', compact('area'));
    }

    /** PUT /products/areas/{area} */
    public function update(Request $request, Area $area)
    {
        $data = $request->validate([
            'name'   => ['required', 'string', 'max:191', Rule::unique('areas', 'name')->ignore($area->id)],
            'status' => ['required', Rule::in(['0','1'])],
        ]);

        $area->update([
            'name'       => $data['name'],
            'status'     => (int)$data['status'],
            'updated_by' => auth()->id() ?? $area->updated_by,
        ]);

        return redirect()->route('products.areas.index')
            ->with('success', 'Area updated successfully.');
    }

    /** DELETE /products/areas/{area} */
    public function destroy(Area $area)
    {
        $area->delete();

        return redirect()->route('products.areas.index')
            ->with('success', 'Area deleted successfully.');
    }
}
