<?php

namespace App\Http;

use Spatie\Menu\Laravel\Html;
use Spatie\Menu\Laravel\Link;
use Spatie\Menu\Laravel\Menu;

/**
 * AdminLTE sidebar renderer using Spatie\Menu.
 *
 * Output:
 * <ul class="sidebar-menu tree" data-widget="tree">
 *   <li class="active"><a href="..."><i class="..."></i> <span>Title</span></a></li>
 *   <li class="treeview active">
 *     <a href="#"><i class="..."></i> <span>Parent</span>
 *        <span class="pull-right-container"><i class="fa fa-angle-left pull-right"></i></span>
 *     </a>
 *     <ul class="treeview-menu">
 *       ...
 *     </ul>
 *   </li>
 * </ul>
 */
class AdminlteSpatieMenu
{
    public static function start(): Menu
    {
        // root <ul class="sidebar-menu tree" data-widget="tree">
        // Spatie\Menu ডিফল্টে <ul> দেয়; আমরা wrapper-এ ক্লাস/অ্যাট্রিবিউট দিচ্ছি
        return Menu::new()
            ->setWrapperTag('ul')
            ->setWrapperAttributes([
                'class' => 'sidebar-menu tree',
                'data-widget' => 'tree',
            ]);
    }

    public static function icon(?string $class): string
    {
        return $class ? "<i class=\"{$class}\"></i> " : '';
    }

    public static function link(string $url, string $title, string $icon = '', bool $active = false)
    {
        $label = Html::raw(self::icon($icon) . '<span>' . e($title) . '</span>');

        $link = Link::to($url, $label)
            ->addParentClass('')    // <li> parent class, চাইলে 'menu-item' দিন
            ->addClass('');         // <a> class

        if ($active) {
            $link->addParentClass('active'); // <li class="active">
        }

        return $link;
    }

    /**
     * AdminLTE dropdown:
     * <li class="treeview [active]">
     *   <a href="#">... + angle-left</a>
     *   <ul class="treeview-menu"> ... </ul>
     * </li>
     */
    public static function dropdown(string $title, string $icon, \Closure $builder, bool $active = false)
    {
        // Parent anchor label with angle-left icon on right
        $titleHtml = Html::raw(
            self::icon($icon)
            . '<span>' . e($title) . '</span>'
            . '<span class="pull-right-container"><i class="fa fa-angle-left pull-right"></i></span>'
        );

        // Child UL
        $child = Menu::new()->addClass('treeview-menu');

        // Let caller add children
        $builder($child);

        // খালি হলে রেন্ডার করব না
        if ($child->isEmpty()) {
            return null;
        }

        // Spatie submenu uses <li> wrapper of parent; আমরা 'treeview' ক্লাস দেব
        $submenu = Menu::new(); // temporary container to create submenu structure
        $submenu->submenu($titleHtml, $child)
            ->each(function ($item) use ($active) {
                // submenu() যে <li> তৈরি করে তাতে আমরা treeview + active দেব
                $item->addParentClass('treeview');
                if ($active) {
                    $item->addParentClass('active');
                }
            });

        // submenu() উপরে ১টা <li> রিটার্ন করে; তাই এটাকে return করলে caller $menu->add(...) করতে পারবে
        // কিন্তু Spatie\Menu\Laravel\Menu::submenu রিটার্ন টাইপ chainable item
        // তাই আমরা caller-কে এই ready-made item return করব
        return $submenu->items()[0] ?? null;
    }
}
